#include "main.h"

GLenum shader_vert;
GLenum shader_frag;
GLenum shader_prog;

Main::Main()
{
	texCoordBufferID = 0;
	gameObjects.clear();
	HUDObjects.clear();
	textureRegistry.clear();
	paletteRegistry.clear();
	audioRegistry.clear();
	newObjectId = 1;

	notDone = true;
	surf_display = NULL;
	screenResolutionX = MAX_GAME_RESOLUTION_X;
	screenResolutionY = MAX_GAME_RESOLUTION_Y;
	gameResolutionX = screenResolutionX;
	gameResolutionY = screenResolutionY;
	fullScreen = false;

	obtainedAudioSpec = NULL;
	currentAudio.clear();

	zoomOutIncrease = false;
	zoomOutDecrease = false;
	zoomOut = 1;
	focusObjectOne = NULL;
	focusObjectTwo = NULL;

	lastFrameTicks = 0;
	frame = 0;

	playerToSetUp = -1;
	currentSettingMapping = SETTING_UP;
	p1GUI = NULL;
	p2GUI = NULL;
	pressButtonForGUI = NULL;
	upGUI = NULL;
	downGUI = NULL;
	leftGUI = NULL;
	rightGUI = NULL;
	jumpGUI = NULL;
	lightAttackGUI = NULL;
	blockGUI = NULL;

	//set up the default input states
	defaultInputs.bKeyUp.held = false; defaultInputs.bKeyUp.pressed = false; defaultInputs.bKeyUp.released = false;
	defaultInputs.bKeyDown.held = false; defaultInputs.bKeyDown.pressed = false; defaultInputs.bKeyDown.released = false;
	defaultInputs.bKeyLeft.held = false; defaultInputs.bKeyLeft.pressed = false; defaultInputs.bKeyLeft.released = false;
	defaultInputs.bKeyRight.held = false; defaultInputs.bKeyRight.pressed = false; defaultInputs.bKeyRight.released = false;
	defaultInputs.bKeyJump.held = false; defaultInputs.bKeyJump.pressed = false; defaultInputs.bKeyJump.released = false;
	defaultInputs.bKeyBlock.held = false; defaultInputs.bKeyBlock.pressed = false; defaultInputs.bKeyBlock.released = false;
	defaultInputs.bKeyLight.held = false; defaultInputs.bKeyLight.pressed = false; defaultInputs.bKeyLight.released = false;
	defaultInputs.bKeyHeavy.held = false; defaultInputs.bKeyHeavy.pressed = false; defaultInputs.bKeyHeavy.released = false;
	defaultInputs.bKeyStart.held = false; defaultInputs.bKeyStart.pressed = false; defaultInputs.bKeyStart.released = false;
	defaultInputs.bKeySelect.held = false; defaultInputs.bKeySelect.pressed = false; defaultInputs.bKeySelect.released = false;
	defaultInputs.bButtonUp.held = false; defaultInputs.bButtonUp.pressed = false; defaultInputs.bButtonUp.released = false;
	defaultInputs.bButtonDown.held = false; defaultInputs.bButtonDown.pressed = false; defaultInputs.bButtonDown.released = false;
	defaultInputs.bButtonLeft.held = false; defaultInputs.bButtonLeft.pressed = false; defaultInputs.bButtonLeft.released = false;
	defaultInputs.bButtonRight.held = false; defaultInputs.bButtonRight.pressed = false; defaultInputs.bButtonRight.released = false;
	defaultInputs.bButtonJump.held = false; defaultInputs.bButtonJump.pressed = false; defaultInputs.bButtonJump.released = false;
	defaultInputs.bButtonBlock.held = false; defaultInputs.bButtonBlock.pressed = false; defaultInputs.bButtonBlock.released = false;
	defaultInputs.bButtonLight.held = false; defaultInputs.bButtonLight.pressed = false; defaultInputs.bButtonLight.released = false;
	defaultInputs.bButtonHeavy.held = false; defaultInputs.bButtonHeavy.pressed = false; defaultInputs.bButtonHeavy.released = false;
	defaultInputs.bButtonStart.held = false; defaultInputs.bButtonStart.pressed = false; defaultInputs.bButtonStart.released = false;
	defaultInputs.bButtonSelect.held = false; defaultInputs.bButtonSelect.pressed = false; defaultInputs.bButtonSelect.released = false;
	defaultInputs.bStickUp.held = false; defaultInputs.bStickUp.pressed = false; defaultInputs.bStickUp.released = false;
	defaultInputs.bStickDown.held = false; defaultInputs.bStickDown.pressed = false; defaultInputs.bStickDown.released = false;
	defaultInputs.bStickLeft.held = false; defaultInputs.bStickLeft.pressed = false; defaultInputs.bStickLeft.released = false;
	defaultInputs.bStickRight.held = false; defaultInputs.bStickRight.pressed = false; defaultInputs.bStickRight.released = false;
	defaultInputs.bStickHardUp.held = false; defaultInputs.bStickHardUp.pressed = false; defaultInputs.bStickHardUp.released = false;
	defaultInputs.bStickHardDown.held = false; defaultInputs.bStickHardDown.pressed = false; defaultInputs.bStickHardDown.released = false;
	defaultInputs.bStickHardLeft.held = false; defaultInputs.bStickHardLeft.pressed = false; defaultInputs.bStickHardLeft.released = false;
	defaultInputs.bStickHardRight.held = false; defaultInputs.bStickHardRight.pressed = false; defaultInputs.bStickHardRight.released = false;
	defaultInputs.bHatUp.held = false; defaultInputs.bHatUp.pressed = false; defaultInputs.bHatUp.released = false;
	defaultInputs.bHatDown.held = false; defaultInputs.bHatDown.pressed = false; defaultInputs.bHatDown.released = false;
	defaultInputs.bHatLeft.held = false; defaultInputs.bHatLeft.pressed = false; defaultInputs.bHatLeft.released = false;
	defaultInputs.bHatRight.held = false; defaultInputs.bHatRight.pressed = false; defaultInputs.bHatRight.released = false;
	defaultInputs.prevInputState = NULL;
	defaultInputs.frame = 0;

	sticks.clear();
	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		players[i] = NULL;
		ClearControls(i);
		InputStates * newHistory = new InputStates();
		*newHistory = defaultInputs;
		inputHistory[i] = newHistory;
		curInputs[i] = new InputStates();
		inputStateChange[i] = false;
		prevJoystickStates[i].hat = SDL_HAT_CENTERED;
		prevJoystickStates[i].stickHeldUp = false;
		prevJoystickStates[i].stickHeldDown = false;
		prevJoystickStates[i].stickHeldLeft = false;
		prevJoystickStates[i].stickHeldRight = false;
		prevJoystickStates[i].stickHeldHardUp = false;
		prevJoystickStates[i].stickHeldHardDown = false;
		prevJoystickStates[i].stickHeldHardLeft = false;
		prevJoystickStates[i].stickHeldHardRight = false;
	}
}

int Main::Execute()
{
	int error = 0;
	if((error = Initialize()) != 0)
	{
		return error;
	}

	UpdateLog("Game initialization complete.", false);

	SDL_Event curEvent;

	UpdateLog("Starting main game loop.", false);

	while(notDone)
	{
		int thisFrameTicks = SDL_GetTicks();
		if(thisFrameTicks - lastFrameTicks >= 1000 / TARGET_FPS)
		{
			lastFrameTicks = thisFrameTicks;
			if(playerToSetUp <= -1)
			{
				if((error = AdvanceHolds()) != 0) { return error; }
			}
			if((error = Event(&curEvent)) != 0) { return error; }
			if(playerToSetUp <= -1)
			{
				if((error = Update()) != 0) { return error; }
				if((error = Collide()) != 0) { return error; }
				if((error = PlayAudio()) != 0) { return error; }
			}
			if((error = Render()) != 0) { return error; }

			frame++;
		}
	}

	UpdateLog("Main game loop exited.", false);

	Cleanup();

	UpdateLog("Cleanup complete.", false);

	return 0;
}

int Main::SetBestGameResolution()
{
	//get the current screen resolution
	const SDL_VideoInfo * vidInfo = SDL_GetVideoInfo();
	screenResolutionX = vidInfo->current_w;
	screenResolutionY = vidInfo->current_h;
	gameResolutionX = MAX_GAME_RESOLUTION_X;
	gameResolutionY = MAX_GAME_RESOLUTION_Y;

	//if the current screen resolution is larger than or equal to the max possible resolution, then just use the max resolution
	if(screenResolutionX >= MAX_GAME_RESOLUTION_X && screenResolutionY >= MAX_GAME_RESOLUTION_Y)
	{
		return 0;
	}

	//pick a viable 16:9 resolution that's equal to or smaller than the screen resolution
	while(gameResolutionX > screenResolutionX || gameResolutionY > screenResolutionY)
	{
		gameResolutionX -= GAME_ASPECT_RATIO_X;
		gameResolutionY -= GAME_ASPECT_RATIO_Y;

		if(gameResolutionX < MIN_GAME_RESOLUTION_X || gameResolutionY < MIN_GAME_RESOLUTION_Y)
		{
			UpdateLog("Screen resolution too small.", true);
			return -1;
		}
	}

	return 0;
}

void AudioCallback(void *unused, Uint8 *stream, int len)
{
	int i;
    Uint32 amount;

	list<CurrentAudioEntry*> toRemove;

	list<CurrentAudioEntry*>::iterator caIt;
	for ( caIt=currentAudio.begin(); caIt != currentAudio.end(); caIt++)
	{
		amount = ((*caIt)->aud->length - (*caIt)->curPosition);
        if ( amount > len )
		{
            amount = len;
        }
		SDL_MixAudio(stream, &(*caIt)->aud->data[(*caIt)->curPosition], amount, SDL_MIX_MAXVOLUME);
		(*caIt)->curPosition += amount;

		if((*caIt)->curPosition >= (*caIt)->aud->length)
		{
			toRemove.push_back((*caIt));
		}
	}

	//remove entries that need to be removed, if any
	for ( caIt=toRemove.begin(); caIt != toRemove.end(); caIt++)
	{
		currentAudio.remove((*caIt));
		delete((*caIt));
	}
}

int Main::Initialize()
{
	int error = 0;
	if((error = SDL_Init(SDL_INIT_EVERYTHING)) != 0)
	{
		return error;
	}

	UpdateLog("SDL Initialized.", false);

	SDL_GL_SetAttribute(SDL_GL_RED_SIZE,        8);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE,      8);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE,       8);
	SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE,      8);
 
	SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE,      16);
	SDL_GL_SetAttribute(SDL_GL_BUFFER_SIZE,        32);
 
	SDL_GL_SetAttribute(SDL_GL_ACCUM_RED_SIZE,    8);
	SDL_GL_SetAttribute(SDL_GL_ACCUM_GREEN_SIZE,    8);
	SDL_GL_SetAttribute(SDL_GL_ACCUM_BLUE_SIZE,    8);
	SDL_GL_SetAttribute(SDL_GL_ACCUM_ALPHA_SIZE,    8);
 
	SDL_GL_SetAttribute(SDL_GL_MULTISAMPLEBUFFERS,  1);
	SDL_GL_SetAttribute(SDL_GL_MULTISAMPLESAMPLES,  2);

	UpdateLog("SDL attributes set.", false);

	//set up SDL screen
	if(int error = SetFullScreen(false) != 0)
	{
		return error;
	}

	////set up audio
	SDL_AudioSpec *desired; 
	SDL_AudioSpec wav_spec; 
	SDL_AudioCVT wav_cvt; 
	Uint32 wav_len; 
	Uint8 *wav_buf; 
	int ret; 

	/* Allocated audio specs */ 
	desired=(SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec)); 
	obtainedAudioSpec=(SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec)); 

	/* Set desired format */ 
	desired->freq=22050; 
	desired->format=AUDIO_S16; 
	desired->channels=1;
	desired->samples=1024; 
	desired->callback=AudioCallback; 
	desired->userdata=NULL; 

	/* Open the audio device */ 
	int blah = SDL_OpenAudio(desired, obtainedAudioSpec);
	if ( blah < 0 )
	{ 
		UpdateLog("Failed to open audio device.", false);
		return -1; //problem opening the audio
	} 
	free(desired); 

	//let audio start running
	SDL_PauseAudio(0);

	//read in definition files
	HSObject * newObject;
	if(int error = LoadDefinition("LOWAS\\LOWAS.xml", &gameObjects, &newObject) != 0)
	{
		return error; //problem loading the definition file
	}
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = 3250;
	newObject->pos.y = 2700;
	if(int error = LoadDefinition("LOWAS\\Left Platform.xml", &gameObjects, &newObject) != 0)
	{
		return error; //problem loading the definition file
	}
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = 1785;
	newObject->pos.y = 3365;
	if(int error = LoadDefinition("LOWAS\\Right Platform.xml", &gameObjects, &newObject) != 0)
	{
		return error; //problem loading the definition file
	}
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = 3838;
	newObject->pos.y = 3036;

	HSObject * fighterOne;
	if(int error = LoadDefinition("John Egbert\\John Egbert.xml", &gameObjects, &fighterOne) != 0)
	{
		return error; //problem loading the definition file
	}
	fighterOne->pos.x = 2315;
	fighterOne->pos.y = 2502;
	players[0] = fighterOne;
	((Fighter*)fighterOne)->state = STANDING;
	((Fighter*)fighterOne)->curHealth = ((Fighter*)fighterOne)->health;
	fighterOne->curHold = ((Fighter*)fighterOne)->fighterEventHolds.standing;
	focusObjectOne = fighterOne;
	
	HSObject * fighterTwo;
	if(int error = LoadDefinition("John Egbert\\John Egbert.xml", &gameObjects, &fighterTwo) != 0)
	{
		return error; //problem loading the definition file
	}
	fighterTwo->pos.x = 4432;
	fighterTwo->pos.y = 2100;
	players[1] = fighterTwo;
	((Fighter*)fighterTwo)->state = STANDING;
	((Fighter*)fighterTwo)->facing = LEFT;
	((Fighter*)fighterTwo)->curHealth = ((Fighter*)fighterTwo)->health;
	fighterTwo->curHold = ((Fighter*)fighterTwo)->fighterEventHolds.standing;
	focusObjectTwo = fighterTwo;

	if(int error = LoadDefinition("Key Config GUI\\playerOne\\playerOne.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	p1GUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\playerTwo\\playerTwo.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	p2GUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\pressButtonFor\\pressButtonFor.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	pressButtonForGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\up\\up.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	upGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\down\\down.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	downGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\left\\left.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	leftGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\right\\right.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	rightGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\jump\\jump.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	jumpGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\lightAttack\\lightAttack.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	lightAttackGUI = newObject;

	if(int error = LoadDefinition("Key Config GUI\\block\\block.xml", &gameObjects, &newObject) != 0) { return error; }
	newObject->curHold = newObject->firstHold;
	newObject->pos.x = KEY_MAP_GUI_AWAY_POS;
	newObject->pos.y = KEY_MAP_GUI_AWAY_POS;
	blockGUI = newObject;

	//HUD setup
	HSObject * newHUD;
	if(int error = LoadDefinition("Test HUD\\Test HUD.xml", &HUDObjects, &newHUD) != 0)
	{
		return error;
	}
	((Fighter*)fighterOne)->hud = (HUD*)newHUD;
	newHUD->curHold = newHUD->firstHold;
	((HUD*)newHUD)->pos.x = (MAX_GAME_RESOLUTION_X / -2) + 20;
	((HUD*)newHUD)->pos.y = (MAX_GAME_RESOLUTION_Y / -2) + 20;
	((HUD*)newHUD)->healthMeterOffset.x = 20;
	((HUD*)newHUD)->healthMeterOffset.y = 20;
	((HUD*)newHUD)->superMeterOffset.x = 20;
	((HUD*)newHUD)->superMeterOffset.y = 90;

	HSObject * newHUDElement;
	if(int error = LoadDefinition("Test HUD\\Test Meter.xml", &HUDObjects, &newHUDElement) != 0)
	{
		return error;
	}
	((HUD*)newHUD)->healthMeter = (Meter*)newHUDElement;
	newHUDElement->curHold = newHUDElement->firstHold;

	if(int error = LoadDefinition("Test HUD\\Test Meter.xml", &HUDObjects, &newHUDElement) != 0)
	{
		return error;
	}
	((HUD*)newHUD)->superMeter = (Meter*)newHUDElement;
	newHUDElement->curHold = newHUDElement->firstHold;
	//end HUD setup

	//HUD setup
	if(int error = LoadDefinition("Test HUD\\Test HUD.xml", &HUDObjects, &newHUD) != 0)
	{
		return error;
	}
	((Fighter*)fighterTwo)->hud = (HUD*)newHUD;
	newHUD->curHold = newHUD->firstHold;
	((HUD*)newHUD)->pos.x = (MAX_GAME_RESOLUTION_X / 2) - 560;
	((HUD*)newHUD)->pos.y = (MAX_GAME_RESOLUTION_Y / -2) + 20;
	((HUD*)newHUD)->healthMeterOffset.x = 20;
	((HUD*)newHUD)->healthMeterOffset.y = 20;
	((HUD*)newHUD)->superMeterOffset.x = 20;
	((HUD*)newHUD)->superMeterOffset.y = 90;

	if(int error = LoadDefinition("Test HUD\\Test Meter.xml", &HUDObjects, &newHUDElement) != 0)
	{
		return error;
	}
	((HUD*)newHUD)->healthMeter = (Meter*)newHUDElement;
	newHUDElement->curHold = newHUDElement->firstHold;

	if(int error = LoadDefinition("Test HUD\\Test Meter.xml", &HUDObjects, &newHUDElement) != 0)
	{
		return error;
	}
	((HUD*)newHUD)->superMeter = (Meter*)newHUDElement;
	newHUDElement->curHold = newHUDElement->firstHold;
	//end HUD setup

	//get joysticks
	for(int i = 0; i < SDL_NumJoysticks(); i++)
	{
		SDL_Joystick * stick = SDL_JoystickOpen(i);
		sticks.push_back(stick);
	}

	//set up some control mappings
	mappings[0].keyUp = SDLK_w;
	mappings[0].keyDown = SDLK_s;
	mappings[0].keyLeft = SDLK_a;
	mappings[0].keyRight = SDLK_d;
	mappings[0].keyJump = SDLK_j;
	mappings[0].keyLight = SDLK_h;
	mappings[0].keyBlock = SDLK_k;

	mappings[1].keyUp = SDLK_UP;
	mappings[1].keyDown = SDLK_DOWN;
	mappings[1].keyLeft = SDLK_LEFT;
	mappings[1].keyRight = SDLK_RIGHT;
	mappings[1].keyJump = SDLK_KP5;
	mappings[1].keyLight = SDLK_KP4;
	mappings[1].keyBlock = SDLK_KP6;

	return 0;
}

int Main::LoadDefinition(string defFilePath, list<HSObject*> * gameObjects, HSObject ** returnValue)
{
	//get the XML structure from the file
	XMLDocument * file = new XMLDocument();
	if(int error = file->LoadFile(defFilePath.data()) != 0)
	{
		stringstream sstm;
		sstm << "Error loading definition file. Code: " << error << " File: " << defFilePath;
		UpdateLog(sstm.str(), true);
		return error; //couldn't load the file
	}

	if(strcmp(file->RootElement()->Value(), "HSObjects") != 0)
	{
		UpdateLog("XML file is not Homestrife definition file: " + defFilePath, true);
		return -1; //this isn't an HSObjects definition file
	}

	//loop through all the object definitions in the xml file
	for(XMLElement * i = file->RootElement()->FirstChildElement(); i != NULL; i = i->NextSiblingElement())
	{
		//get the object type
		const char * objectType = i->Value();

		//create a new object of the appropriate type
		HSObject * newObject;
		if(strcmp(objectType, "HSObject") == 0)
		{
			newObject = new HSObject();
		}
		else if(strcmp(objectType, "TerrainObject") == 0)
		{
			newObject = new TerrainObject();
		}
		else if(strcmp(objectType, "PhysicsObject") == 0)
		{
			newObject = new PhysicsObject();
		}
		else if(strcmp(objectType, "Fighter") == 0)
		{
			newObject = new Fighter();
		}
		else if(strcmp(objectType, "HUD") == 0)
		{
			newObject = new HUD();
		}
		else if(strcmp(objectType, "Meter") == 0)
		{
			newObject = new Meter();
		}
		else if(strcmp(objectType, "Counter") == 0)
		{
			newObject = new Counter();
		}
		else
		{
			continue; //invalid object type
		}

		//execute the object's local definition code
		string defFileDirectory = defFilePath.substr(0, defFilePath.find_last_of("\\"));
		if(defFileDirectory.compare(defFilePath) == 0)
		{
			defFileDirectory = "";
		}
		string currentWorkingDirectory = GetCurrentWorkingDirectory(NULL, 0);
		defFileDirectory = currentWorkingDirectory + "\\" + defFileDirectory;
		if(int error = newObject->Define(i, defFileDirectory, &textureRegistry, &paletteRegistry, &audioRegistry, obtainedAudioSpec) != 0)
		{
			return error; //there was an error defining the object
		}

		//give the new object a unique id
		newObject->id = newObjectId;
		newObjectId++;

		//add the object to the main list of objects
		gameObjects->push_back(newObject);

		//save the last object created as the return value
		if(returnValue != NULL)
		{
			*returnValue = newObject;
		}
	}

	delete file;

	return 0;
}

int Main::Update()
{
	list<HSObject*>::iterator objIt;
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		if(!(*objIt)->IsTerrain() && (*objIt)->pos.y > 6000)
		{
			(*objIt)->pos.x = 3200;
			(*objIt)->pos.y = 2000;
		}

		if(int error = (*objIt)->Update() != 0)
		{
			return error; //there was an error updating the object
		}
	}
	
	for ( objIt=HUDObjects.begin(); objIt != HUDObjects.end(); objIt++)
	{
		if(int error = (*objIt)->Update() != 0)
		{
			return error; //there was an error updating the object
		}
	}

	return 0;
}

int Main::Collide()
{
	//first handle all terrain box collision
	list<HSObject*>::iterator objIt;
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		//pass the location of the list of game objects into each object's CollideTerrain function
		if(int error = (*objIt)->CollideTerrain(&gameObjects) != 0)
		{
			return error; //there was an error doing terrain collisions
		}
	}
	
	//now, handle all attack/hurt box collision
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		//pass the location of the list of game objects into each object's CollideAttack function
		if(int error = (*objIt)->CollideAttack(&gameObjects) != 0)
		{
			return error; //there was an error doing Hit/Hurt collisions
		}
	}

	//finally, apply all the results of the attack collision phase
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		(*objIt)->ApplyAttackResults();
	}

	return 0;
}

int Main::Render()
{
	//adjust the camera
 
	//get the current screen resolution
	if(zoomOutIncrease) { zoomOut += ZOOM_STEP; }
	else if(zoomOutDecrease) { zoomOut -= ZOOM_STEP; }
	if(zoomOut < 1) { zoomOut = 1; }

	HSVect2D focusPos;
	focusPos.x = MAX_GAME_RESOLUTION_X;
	focusPos.y = MAX_GAME_RESOLUTION_Y;
	if(focusObjectOne != NULL && focusObjectTwo != NULL)
	{
		//get the location between the two objects
		HSVect2D posActualOne;
		posActualOne.x = focusObjectOne->pos.x;
		posActualOne.y = focusObjectOne->pos.y;
		HSVect2D posActualTwo;
		posActualTwo.x = focusObjectTwo->pos.x;
		posActualTwo.y = focusObjectTwo->pos.y;

		if(focusObjectOne->IsFighter())
		{
			Fighter * f = (Fighter *)focusObjectOne;
			posActualOne.x = posActualOne.x + f->firstUprightTerrainBox->offset.x + f->firstUprightTerrainBox->width / 2;
			posActualOne.y = posActualOne.y + f->firstUprightTerrainBox->offset.y + f->firstUprightTerrainBox->height / 2;
		}

		if(focusObjectTwo->IsFighter())
		{
			Fighter * f = (Fighter *)focusObjectTwo;
			posActualTwo.x = posActualTwo.x + f->firstUprightTerrainBox->offset.x + f->firstUprightTerrainBox->width / 2;
			posActualTwo.y = posActualTwo.y + f->firstUprightTerrainBox->offset.y + f->firstUprightTerrainBox->height / 2;
		}

		HSVect2D posDiff;
		posDiff.x = posActualTwo.x - posActualOne.x;
		posDiff.y = posActualTwo.y - posActualOne.y;

		focusPos.x = posActualOne.x + posDiff.x / 2;
		focusPos.y = posActualOne.y + posDiff.y / 2;

		//get the zoom
		float xZoomOut = 1;
		float yZoomOut = 1;
		if(abs(posDiff.x) + ZOOM_BOUNDARY_X_THRESHOLD * 2 > MAX_GAME_RESOLUTION_X)
		{
			float minRes = MAX_GAME_RESOLUTION_X;
			float neededRes = abs(posDiff.x) + ZOOM_BOUNDARY_X_THRESHOLD * 2;
			xZoomOut = neededRes / minRes;
		}
		if(abs(posDiff.y) + ZOOM_BOUNDARY_Y_THRESHOLD * 2 > MAX_GAME_RESOLUTION_Y)
		{
			float minRes = MAX_GAME_RESOLUTION_Y;
			float neededRes = abs(posDiff.y) + ZOOM_BOUNDARY_Y_THRESHOLD * 2;
			yZoomOut = neededRes / minRes;
		}
		if(xZoomOut > yZoomOut) { zoomOut = xZoomOut; }
		else { zoomOut = yZoomOut; }

		if(zoomOut < 1) { zoomOut = 1; }
	}
	else if(focusObjectOne != NULL)
	{
		focusPos.x = focusObjectOne->pos.x;
		focusPos.y = focusObjectOne->pos.y;

		if(focusObjectOne->IsFighter())
		{
			//get the centerpoint of upright terrain box
			Fighter * f = (Fighter *)focusObjectOne;
			focusPos.x = focusPos.x + f->firstUprightTerrainBox->offset.x + f->firstUprightTerrainBox->width / 2;
			focusPos.y = focusPos.y + f->firstUprightTerrainBox->offset.y + f->firstUprightTerrainBox->height / 2;
		}
	}

	double left = focusPos.x - ((MAX_GAME_RESOLUTION_X * zoomOut) / 2);
	double right = focusPos.x + ((MAX_GAME_RESOLUTION_X * zoomOut) / 2);
	double bottom = focusPos.y + ((MAX_GAME_RESOLUTION_Y * zoomOut) / 2);
	double top = focusPos.y - ((MAX_GAME_RESOLUTION_Y * zoomOut) / 2);

	//position the key map gui
	if(playerToSetUp > -1)
	{
		if(playerToSetUp == 0)
		{
			p1GUI->pos.x = left + KEY_MAP_GUI_POS_X;
			p1GUI->pos.y = top + P1_P2_POS_Y;
		}
		else if(playerToSetUp == 1)
		{
			p2GUI->pos.x = left + KEY_MAP_GUI_POS_X;
			p2GUI->pos.y = top + P1_P2_POS_Y;
		}

		pressButtonForGUI->pos.x = left + KEY_MAP_GUI_POS_X;
		pressButtonForGUI->pos.y = top + PRESS_BUTTON_FOR_POS_Y;

		switch(currentSettingMapping)
		{
		case SETTING_UP: upGUI->pos.x = left + KEY_MAP_GUI_POS_X; upGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_DOWN: downGUI->pos.x = left + KEY_MAP_GUI_POS_X; downGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_LEFT: leftGUI->pos.x = left + KEY_MAP_GUI_POS_X; leftGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_RIGHT: rightGUI->pos.x = left + KEY_MAP_GUI_POS_X; rightGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_JUMP: jumpGUI->pos.x = left + KEY_MAP_GUI_POS_X; jumpGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_LIGHT: lightAttackGUI->pos.x = left + KEY_MAP_GUI_POS_X; lightAttackGUI->pos.y = top + BUTTON_POS_Y; break;
		case SETTING_BLOCK: blockGUI->pos.x = left + KEY_MAP_GUI_POS_X; blockGUI->pos.y = top + BUTTON_POS_Y; break;
		}
	}

	//render objects
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(left, right, bottom, top, 1, -1);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	int uniformTexLocation = glGetUniformLocationARB(shader_prog, "tex");
	int uniformIndexedLocation = glGetUniformLocationARB(shader_prog, "indexed");
	int uniformIndexTexLocation = glGetUniformLocationARB(shader_prog, "indexTex");
	int uniformPaletteLocation = glGetUniformLocationARB(shader_prog, "palette");

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	list<HSObject*>::iterator objIt;
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		list<TextureInstance>::iterator texIt;
		for ( texIt=(*objIt)->curHold->textures.begin(); texIt != (*objIt)->curHold->textures.end(); texIt++)
		{
			RenderTexture((*objIt), (*texIt), uniformTexLocation, uniformIndexedLocation, uniformIndexTexLocation, uniformPaletteLocation);
		}
	}

	left = MAX_GAME_RESOLUTION_X / -2;
	right = MAX_GAME_RESOLUTION_X / 2;
	bottom = MAX_GAME_RESOLUTION_Y / 2;
	top = MAX_GAME_RESOLUTION_Y / -2;

	//render HUD
	glPushMatrix();
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(left, right, bottom, top, 1, -1);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	for ( objIt=HUDObjects.begin(); objIt != HUDObjects.end(); objIt++)
	{
		list<TextureInstance>::iterator texIt;
		for ( texIt=(*objIt)->curHold->textures.begin(); texIt != (*objIt)->curHold->textures.end(); texIt++)
		{
			RenderTexture((*objIt), (*texIt), uniformTexLocation, uniformIndexedLocation, uniformIndexTexLocation, uniformPaletteLocation);
		}
	}

	glPopMatrix();

	glBindTexture(GL_TEXTURE_2D, 0);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	SDL_GL_SwapBuffers();

	return 0;
}

int Main::RenderTexture(HSObject * obj, TextureInstance tex, int uTexLoc, int uIndLoc, int uIndTexLoc, int uPalLoc)
{
	//get the texture and fragment shader inputs set up
	glActiveTextureARB(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, tex.hsTex->textureID);
	if(tex.hsTex->indexed)
	{
		glUniform1iARB(uIndTexLoc, 0);
		glActiveTextureARB(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D, obj->palette->textureID);
		glPixelStorei(GL_PACK_ALIGNMENT, 1);
		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		glActiveTextureARB(GL_TEXTURE0);
		glUniform1iARB(uPalLoc, 1);
		glUniform1iARB(uIndLoc, 1);
		glPixelStorei(GL_PACK_ALIGNMENT, 1);
		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	}
	else
	{
		glUniform1iARB(uTexLoc, 0);
		glUniform1iARB(uIndLoc, 0);
		glPixelStorei(GL_PACK_ALIGNMENT, 4);
		glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
	}

	//get the actual texture offset, based on the object's current hFlip
	float hScale = tex.hScale;
	float vScale = tex.vScale;
	if(obj->hFlip)
	{
		hScale *= -1;
	}

	//load the identity matrix and do translations
	glLoadIdentity();
	glTranslatef(obj->pos.x + tex.offset.x * hScale, obj->pos.y + tex.offset.y * vScale, 0);
	glScalef(hScale, vScale, 1);

	//get the vertex array set up
	glBindBuffer(GL_ARRAY_BUFFER, tex.hsTex->bufferID);
	glVertexPointer(3, GL_SHORT, 0, (char*) NULL);

	//get the texture coordinate array set up
	glBindBuffer(GL_ARRAY_BUFFER, texCoordBufferID);
	glTexCoordPointer(2, GL_FLOAT, 0, (char*) NULL);

	//draw stuff
	glDrawArrays(GL_QUADS, 0, 4);

	return 0;
}

int Main::PlayAudio()
{
	list<HSObject*>::iterator objIt;
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		list<HSAudio*> audio = (*objIt)->GetAudio();

		list<HSAudio*>::iterator audIt;
		for ( audIt=audio.begin(); audIt != audio.end(); audIt++)
		{
			CurrentAudioEntry * newEntry = new CurrentAudioEntry();
			newEntry->aud = (*audIt);
			newEntry->curPosition = 0;

			SDL_LockAudio();
			currentAudio.push_back(newEntry);
			SDL_UnlockAudio();
		}
	}

	return 0;
}

int Main::Cleanup()
{
	glDeleteObjectARB(shader_vert);
	glDeleteObjectARB(shader_frag);
	glDeleteObjectARB(shader_prog);

	list<HSObject*>::iterator it;
	for ( it=gameObjects.begin(); it != gameObjects.end(); it++)
	{
		delete(*it);
	}
	gameObjects.clear();

	list<SDL_Joystick*>::iterator it2;
	for ( it2=sticks.begin(); it2 != sticks.end(); it2++)
	{
		SDL_JoystickClose(*it2);
	}
	sticks.clear();

	SDL_Quit();

	return 0;
}

//animation and holds
int Main::AdvanceHolds()
{
	list<HSObject*>::iterator objIt;
	for ( objIt=gameObjects.begin(); objIt != gameObjects.end(); objIt++)
	{
		if(int error = (*objIt)->AdvanceHolds() != 0)
		{
			return error; //there was an error advancing holds
		}
	}
	
	for ( objIt=HUDObjects.begin(); objIt != HUDObjects.end(); objIt++)
	{
		if(int error = (*objIt)->AdvanceHolds() != 0)
		{
			return error; //there was an error advancing holds
		}
	}

	return 0;
}

//event/input stuff
int Main::Event(SDL_Event* curEvent)
{
	//reset current inputs
	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		inputStateChange[i] = false;
		*curInputs[i] = defaultInputs;
		curInputs[i]->frame = frame;
	}

	//get current inputs
	while(SDL_PollEvent(curEvent))
	{
        HandleEvent(curEvent);
    }
	
	if(playerToSetUp > -1) { return 0; }

	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(!inputStateChange[i]) { continue; }

		//set "held" for keyboard key presses
		curInputs[i]->bKeyUp.held = IsHeld(&curInputs[i]->bKeyUp, &inputHistory[i]->bKeyUp);
		curInputs[i]->bKeyDown.held = IsHeld(&curInputs[i]->bKeyDown, &inputHistory[i]->bKeyDown);
		curInputs[i]->bKeyLeft.held = IsHeld(&curInputs[i]->bKeyLeft, &inputHistory[i]->bKeyLeft);
		curInputs[i]->bKeyRight.held = IsHeld(&curInputs[i]->bKeyRight, &inputHistory[i]->bKeyRight);
		curInputs[i]->bKeyJump.held = IsHeld(&curInputs[i]->bKeyJump, &inputHistory[i]->bKeyJump);
		curInputs[i]->bKeyBlock.held = IsHeld(&curInputs[i]->bKeyBlock, &inputHistory[i]->bKeyBlock);
		curInputs[i]->bKeyLight.held = IsHeld(&curInputs[i]->bKeyLight, &inputHistory[i]->bKeyLight);
		curInputs[i]->bKeyHeavy.held = IsHeld(&curInputs[i]->bKeyHeavy, &inputHistory[i]->bKeyHeavy);
		curInputs[i]->bKeyStart.held = IsHeld(&curInputs[i]->bKeyStart, &inputHistory[i]->bKeyStart);
		curInputs[i]->bKeySelect.held = IsHeld(&curInputs[i]->bKeySelect, &inputHistory[i]->bKeySelect);
		
		//set "held" for joystick button presses
		curInputs[i]->bButtonUp.held = IsHeld(&curInputs[i]->bButtonUp, &inputHistory[i]->bButtonUp);
		curInputs[i]->bButtonDown.held = IsHeld(&curInputs[i]->bButtonDown, &inputHistory[i]->bButtonDown);
		curInputs[i]->bButtonLeft.held = IsHeld(&curInputs[i]->bButtonLeft, &inputHistory[i]->bButtonLeft);
		curInputs[i]->bButtonRight.held = IsHeld(&curInputs[i]->bButtonRight, &inputHistory[i]->bButtonRight);
		curInputs[i]->bButtonJump.held = IsHeld(&curInputs[i]->bButtonJump, &inputHistory[i]->bButtonJump);
		curInputs[i]->bButtonBlock.held = IsHeld(&curInputs[i]->bButtonBlock, &inputHistory[i]->bButtonBlock);
		curInputs[i]->bButtonLight.held = IsHeld(&curInputs[i]->bButtonLight, &inputHistory[i]->bButtonLight);
		curInputs[i]->bButtonHeavy.held = IsHeld(&curInputs[i]->bButtonHeavy, &inputHistory[i]->bButtonHeavy);
		curInputs[i]->bButtonStart.held = IsHeld(&curInputs[i]->bButtonStart, &inputHistory[i]->bButtonStart);
		curInputs[i]->bButtonSelect.held = IsHeld(&curInputs[i]->bButtonSelect, &inputHistory[i]->bButtonSelect);

		//set "held" for joystick hat presses
		curInputs[i]->bHatUp.held = IsHeld(&curInputs[i]->bHatUp, &inputHistory[i]->bHatUp);
		curInputs[i]->bHatDown.held = IsHeld(&curInputs[i]->bHatDown, &inputHistory[i]->bHatDown);
		curInputs[i]->bHatLeft.held = IsHeld(&curInputs[i]->bHatLeft, &inputHistory[i]->bHatLeft);
		curInputs[i]->bHatRight.held = IsHeld(&curInputs[i]->bHatRight, &inputHistory[i]->bHatRight);

		//set "held" for joystick stick movements
		curInputs[i]->bStickUp.held = IsHeld(&curInputs[i]->bStickUp, &inputHistory[i]->bStickUp);
		curInputs[i]->bStickDown.held = IsHeld(&curInputs[i]->bStickDown, &inputHistory[i]->bStickDown);
		curInputs[i]->bStickLeft.held = IsHeld(&curInputs[i]->bStickLeft, &inputHistory[i]->bStickLeft);
		curInputs[i]->bStickRight.held = IsHeld(&curInputs[i]->bStickRight, &inputHistory[i]->bStickRight);
		curInputs[i]->bStickHardUp.held = IsHeld(&curInputs[i]->bStickHardUp, &inputHistory[i]->bStickHardUp);
		curInputs[i]->bStickHardDown.held = IsHeld(&curInputs[i]->bStickHardDown, &inputHistory[i]->bStickHardDown);
		curInputs[i]->bStickHardLeft.held = IsHeld(&curInputs[i]->bStickHardLeft, &inputHistory[i]->bStickHardLeft);
		curInputs[i]->bStickHardRight.held = IsHeld(&curInputs[i]->bStickHardRight, &inputHistory[i]->bStickHardRight);

		//save the current inputs
		InputStates * newInputs = new InputStates();
		*newInputs = *curInputs[i];
		newInputs->prevInputState = inputHistory[i];
		inputHistory[i] = newInputs;

		//clean out the outdated input history
		/*int historyCount = 1;
		InputStates * is = inputHistory[i]->prevInputStates;
		InputStates * prevIs = NULL;
		while((is = is->prevInputStates) != NULL)
		{
			delete prevIs;
			historyCount++;
			if(historyCount > MAX_INPUT_HISTORY)
			{
				prevIs = is;
			}
		}

		delete prevIs;*/
	}
	
	//call event handling on players
	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(players[i] != NULL)
		{
			players[i]->Event(inputHistory[i], frame);
		}
	}

	return 0;
}

bool Main::IsHeld(InputState * cur, InputState * prev)
{
	//set "held" to true if "pressed" is true OR if the previous inputs "held" true but the current "released" is not true
	if(cur->pressed == true || (cur->released == false && prev->held == true))
	{
		return true;
	}
	return false;
}

bool Main::IsPressed(InputState * cur, InputState * prev)
{
	//if "held" is true currently but wasn't previously, set "pressed" to true
	if(cur->held == true && prev->held == false)
	{
		return true;
	}
	return false;
}

bool Main::IsReleased(InputState * cur, InputState * prev)
{
	//if "held" is currently not true but was true perviously, set "released" to true
	if(cur->held == false && prev->held == true)
	{
		return true;
	}
	return false;
}

void Main::ClearControls(int player)
{
	if(player < 0 || player >= MAX_PLAYERS) { return; }
	mappings[player].buttonUp.button = 1000;
	mappings[player].buttonUp.joystick = 1000;
	mappings[player].buttonDown.button = 1000;
	mappings[player].buttonDown.joystick = 1000;
	mappings[player].buttonLeft.button = 1000;
	mappings[player].buttonLeft.joystick = 1000;
	mappings[player].buttonRight.button = 1000;
	mappings[player].buttonRight.joystick = 1000;
	mappings[player].buttonJump.button = 1000;
	mappings[player].buttonJump.joystick = 1000;
	mappings[player].buttonLight.button = 1000;
	mappings[player].buttonLight.joystick = 1000;
	mappings[player].buttonHeavy.button = 1000;
	mappings[player].buttonHeavy.joystick = 1000;
	mappings[player].buttonBlock.button = 1000;
	mappings[player].buttonBlock.joystick = 1000;
	mappings[player].buttonStart.button = 1000;
	mappings[player].buttonStart.joystick = 1000;
	mappings[player].buttonSelect.button = 1000;
	mappings[player].buttonSelect.joystick = 1000;

	mappings[player].keyUp = SDLK_UNKNOWN;
	mappings[player].keyDown = SDLK_UNKNOWN;
	mappings[player].keyLeft = SDLK_UNKNOWN;
	mappings[player].keyRight = SDLK_UNKNOWN;
	mappings[player].keyJump = SDLK_UNKNOWN;
	mappings[player].keyLight = SDLK_UNKNOWN;
	mappings[player].keyHeavy = SDLK_UNKNOWN;
	mappings[player].keyBlock = SDLK_UNKNOWN;
	mappings[player].keyStart = SDLK_UNKNOWN;
	mappings[player].keySelect = SDLK_UNKNOWN;

	mappings[player].hat = 1000;
	mappings[player].stick = 1000;
}

void Main::SetControls(int player)
{
	ClearControls(player);

	playerToSetUp = player;
	currentSettingMapping = SETTING_UP;
}

void Main::NextControl(bool hatStickSet)
{
	if(hatStickSet && (currentSettingMapping == SETTING_UP || currentSettingMapping == SETTING_DOWN ||
			currentSettingMapping == SETTING_LEFT || currentSettingMapping == SETTING_RIGHT))
	{
		currentSettingMapping = SETTING_JUMP;
	}
	else
	{
		if(currentSettingMapping == SETTING_BLOCK) { DoneSettingControls(); return; }
		else { currentSettingMapping = (CurrentSettingMapping)(currentSettingMapping + 1); }
	}

	p1GUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	p1GUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	p2GUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	p2GUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	pressButtonForGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	pressButtonForGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	upGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	upGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	downGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	downGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	leftGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	leftGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	rightGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	rightGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	jumpGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	jumpGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	lightAttackGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	lightAttackGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	blockGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	blockGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
}

void Main::DoneSettingControls()
{
	playerToSetUp = -1;

	p1GUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	p1GUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	p2GUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	p2GUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	pressButtonForGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	pressButtonForGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	upGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	upGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	downGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	downGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	leftGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	leftGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	rightGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	rightGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	jumpGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	jumpGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	lightAttackGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	lightAttackGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
	blockGUI->pos.x = KEY_MAP_GUI_AWAY_POS;
	blockGUI->pos.y = KEY_MAP_GUI_AWAY_POS;
}

void Main::ChangePlayerPalette(int player)
{
	HSPalette * curPalette = players[player]->palette;
	players[player]->palette = NULL;

	curPalette->usingCount--;
	if(curPalette->usingCount < 0) { curPalette->usingCount = 0; }

	list<HSPalette*>::iterator plIt;
	bool getNextPalette = false;
	for ( plIt=players[player]->palettes.begin(); plIt != players[player]->palettes.end(); plIt++)
	{
		if(getNextPalette && (*plIt)->usingCount <= 0)
		{
			players[player]->palette = (*plIt);
			(*plIt)->usingCount++;
			break;
		}
		else if((*plIt) == curPalette)
		{
			getNextPalette = true;
		}
	}
	
	if(players[player]->palette == NULL && !players[player]->palettes.empty())
	{
		players[player]->palette = players[player]->palettes.front();
		players[player]->palette->usingCount++;
	}
}

int Main::HandleEvent(SDL_Event* Event) 
{
    switch(Event->type) 
	{
        case SDL_ACTIVEEVENT: 
		{
            switch(Event->active.state) 
			{
                case SDL_APPMOUSEFOCUS: 
				{
                    if ( Event->active.gain )  MouseFocus();
                    else               MouseBlur();

                    break;
                }
                case SDL_APPINPUTFOCUS: 
				{
                    if ( Event->active.gain )  InputFocus();
                    else               InputBlur();

                    break;
                }
                case SDL_APPACTIVE: 
				{
                    if ( Event->active.gain )  Restore();
                    else               Minimize();

                    break;
                }
            }
            break;
        }

        case SDL_KEYDOWN: 
		{
			KeyDown(Event->key.keysym.sym,Event->key.keysym.mod,Event->key.keysym.unicode);
            break;
        }

        case SDL_KEYUP: 
		{
			KeyUp(Event->key.keysym.sym,Event->key.keysym.mod,Event->key.keysym.unicode);
            break;
        }

        case SDL_MOUSEMOTION: 
		{
			MouseMove(Event->motion.x,Event->motion.y,Event->motion.xrel,Event->motion.yrel,(Event->motion.state&SDL_BUTTON(SDL_BUTTON_LEFT))!=0,(Event->motion.state&SDL_BUTTON(SDL_BUTTON_RIGHT))!=0,(Event->motion.state&SDL_BUTTON(SDL_BUTTON_MIDDLE))!=0);
            break;
        }

        case SDL_MOUSEBUTTONDOWN: 
			{
            switch(Event->button.button) 
			{
                case SDL_BUTTON_LEFT: 
				{
					LButtonDown(Event->button.x,Event->button.y);
                    break;
                }
                case SDL_BUTTON_RIGHT: 
				{
					RButtonDown(Event->button.x,Event->button.y);
                    break;
                }
                case SDL_BUTTON_MIDDLE: 
				{
					MButtonDown(Event->button.x,Event->button.y);
                    break;
                }
            }
            break;
        }

        case SDL_MOUSEBUTTONUP: 
			{
            switch(Event->button.button) 
			{
                case SDL_BUTTON_LEFT: 
				{
					LButtonUp(Event->button.x,Event->button.y);
                    break;
                }
                case SDL_BUTTON_RIGHT: 
				{
					RButtonUp(Event->button.x,Event->button.y);
                    break;
                }
                case SDL_BUTTON_MIDDLE: 
				{
					MButtonUp(Event->button.x,Event->button.y);
                    break;
                }
            }
            break;
        }

        case SDL_JOYAXISMOTION: 
		{
			JoyAxis(Event->jaxis.which,Event->jaxis.axis,Event->jaxis.value);
            break;
        }

        case SDL_JOYBALLMOTION: 
		{
			JoyBall(Event->jball.which,Event->jball.ball,Event->jball.xrel,Event->jball.yrel);
            break;
        }

        case SDL_JOYHATMOTION: 
		{
			JoyHat(Event->jhat.which,Event->jhat.hat,Event->jhat.value);
            break;
        }
        case SDL_JOYBUTTONDOWN:
		{
			JoyButtonDown(Event->jbutton.which,Event->jbutton.button);
            break;
        }

        case SDL_JOYBUTTONUP:
		{
			JoyButtonUp(Event->jbutton.which,Event->jbutton.button);
            break;
        }

        case SDL_QUIT: 
		{
			Exit();
            break;
        }

        case SDL_SYSWMEVENT: 
		{
            //Ignore
            break;
        }

        case SDL_VIDEORESIZE: 
		{
			Resize(Event->resize.w,Event->resize.h);
            break;
        }

        case SDL_VIDEOEXPOSE: 
		{
			Expose();
            break;
        }

        default: 
		{
			User(Event->user.type,Event->user.code,Event->user.data1,Event->user.data2);
            break;
        }
    }

	return 0;
}



void Main::InputFocus() 
{
    
}

void Main::InputBlur() 
{
    
}

void Main::KeyDown(SDLKey sym, SDLMod mod, Uint16 unicode) 
{
	if(sym == SDLK_ESCAPE)
	{
		Exit();
		return;
	}
	else if(sym == SDLK_0)
	{
		ToggleFullScreen();
		return;
	}
	else if(sym == SDLK_1)
	{
		SetControls(0);
		return;
	}
	else if(sym == SDLK_2)
	{
		SetControls(1);
		return;
	}
	else if(sym == SDLK_5)
	{
		ChangePlayerPalette(0);
		return;
	}
	else if(sym == SDLK_6)
	{
		ChangePlayerPalette(1);
		return;
	}

	if(playerToSetUp > -1)
	{
		switch(currentSettingMapping)
		{
			case SETTING_UP: mappings[playerToSetUp].keyUp = sym; break;
			case SETTING_DOWN: mappings[playerToSetUp].keyDown = sym; break;
			case SETTING_LEFT: mappings[playerToSetUp].keyLeft = sym; break;
			case SETTING_RIGHT: mappings[playerToSetUp].keyRight = sym; break;
			case SETTING_JUMP: mappings[playerToSetUp].keyJump = sym; break;
			case SETTING_LIGHT: mappings[playerToSetUp].keyLight = sym; break;
			case SETTING_BLOCK: mappings[playerToSetUp].keyBlock = sym; break;
		}

		NextControl(false);

		return;
	}

	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].keyUp == sym)
		{
			curInputs[i]->bKeyUp.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyDown == sym)
		{
			curInputs[i]->bKeyDown.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyLeft == sym)
		{
			curInputs[i]->bKeyLeft.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyRight == sym)
		{
			curInputs[i]->bKeyRight.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyJump == sym)
		{
			curInputs[i]->bKeyJump.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyBlock == sym)
		{
			curInputs[i]->bKeyBlock.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyLight == sym)
		{
			curInputs[i]->bKeyLight.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyHeavy == sym)
		{
			curInputs[i]->bKeyHeavy.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyStart == sym)
		{
			curInputs[i]->bKeyStart.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keySelect == sym)
		{
			curInputs[i]->bKeySelect.pressed = true;
			inputStateChange[i] = true;
		}
	}
}

void Main::KeyUp(SDLKey sym, SDLMod mod, Uint16 unicode) 
{
	if(playerToSetUp > -1)
	{
		return;
	}

	if(sym == SDLK_MINUS)
	{
		zoomOutIncrease = false;
	}
	else if(sym == SDLK_EQUALS)
	{
		zoomOutDecrease = false;
	}

	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].keyUp == sym)
		{
			curInputs[i]->bKeyUp.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyDown == sym)
		{
			curInputs[i]->bKeyDown.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyLeft == sym)
		{
			curInputs[i]->bKeyLeft.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyRight == sym)
		{
			curInputs[i]->bKeyRight.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyJump == sym)
		{
			curInputs[i]->bKeyJump.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyBlock == sym)
		{
			curInputs[i]->bKeyBlock.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyLight == sym)
		{
			curInputs[i]->bKeyLight.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyHeavy == sym)
		{
			curInputs[i]->bKeyHeavy.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keyStart == sym)
		{
			curInputs[i]->bKeyStart.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].keySelect == sym)
		{
			curInputs[i]->bKeySelect.released = true;
			inputStateChange[i] = true;
		}
	}
}

void Main::MouseFocus() 
{

}

void Main::MouseBlur() 
{
    
}

void Main::MouseMove(int mX, int mY, int relX, int relY, bool Left,bool Right,bool Middle) 
{
    
}

void Main::MouseWheel(bool Up, bool Down) 
{
    
}

void Main::LButtonDown(int mX, int mY) 
{
    
}

void Main::LButtonUp(int mX, int mY) 
{
    
}

void Main::RButtonDown(int mX, int mY) 
{
    
}

void Main::RButtonUp(int mX, int mY) 
{
    
}

void Main::MButtonDown(int mX, int mY) 
{
    
}

void Main::MButtonUp(int mX, int mY) 
{
    
}

void Main::JoyAxis(Uint8 which,Uint8 axis,Sint16 value) 
{
	if(playerToSetUp > -1)
	{
		if((currentSettingMapping == SETTING_UP || currentSettingMapping == SETTING_DOWN ||
			currentSettingMapping == SETTING_LEFT || currentSettingMapping == SETTING_RIGHT) &&
			abs(value) > STICK_THRESHOLD)
		{
			ClearControls(playerToSetUp);
			mappings[playerToSetUp].stick = which;

			NextControl(true);
		}

		return;
	}

    InputStates * tempInputs = NULL;
	PreviousJoystickStates * prevStickState = NULL;
	int player = -1;
	
	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].stick == which)
		{
			tempInputs = curInputs[i];
			prevStickState = &prevJoystickStates[i];
			player = i;
		}
	}

	if(tempInputs == NULL) { return; }

	if(axis == 0)
	{
		if(value < -STICK_THRESHOLD)
		{
			if(!prevStickState->stickHeldLeft) { tempInputs->bStickLeft.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldLeft = true;
		}
		else if(value > STICK_THRESHOLD)
		{
			if(!prevStickState->stickHeldRight) { tempInputs->bStickRight.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldRight = true;
		}
		else
		{
			if(prevStickState->stickHeldLeft) { tempInputs->bStickLeft.released = true; inputStateChange[player] = true; }
			if(prevStickState->stickHeldRight) { tempInputs->bStickRight.released = true; inputStateChange[player] = true; }
			prevStickState->stickHeldLeft = false;
			prevStickState->stickHeldRight = false;
		}

		if(value < -STICK_HARD_THRESHOLD)
		{
			if(!prevStickState->stickHeldHardLeft) { tempInputs->bStickHardLeft.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardLeft = true;
		}
		else if(value > STICK_HARD_THRESHOLD)
		{
			if(!prevStickState->stickHeldHardRight) { tempInputs->bStickHardRight.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardRight = true;
		}
		else
		{
			if(prevStickState->stickHeldHardLeft) { tempInputs->bStickHardLeft.released = true; inputStateChange[player] = true; }
			if(prevStickState->stickHeldHardRight) { tempInputs->bStickHardRight.released = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardLeft = false;
			prevStickState->stickHeldHardRight = false;
		}
	}
	else if(axis == 1)
	{
		if(value < -STICK_THRESHOLD)
		{
			if(!prevStickState->stickHeldUp) { tempInputs->bStickUp.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldUp = true;
		}
		else if(value > STICK_THRESHOLD)
		{
			if(!prevStickState->stickHeldDown) { tempInputs->bStickDown.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldDown = true;
		}
		else
		{
			if(prevStickState->stickHeldUp) { tempInputs->bStickUp.released = true; inputStateChange[player] = true; }
			if(prevStickState->stickHeldDown) { tempInputs->bStickDown.released = true; inputStateChange[player] = true; }
			prevStickState->stickHeldUp = false;
			prevStickState->stickHeldDown = false;
		}
		
		if(value < -STICK_HARD_THRESHOLD)
		{
			if(!prevStickState->stickHeldHardUp) { tempInputs->bStickHardUp.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardUp = true;
		}
		else if(value > STICK_HARD_THRESHOLD)
		{
			if(!prevStickState->stickHeldHardDown) { tempInputs->bStickHardDown.pressed = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardDown = true;
		}
		else
		{
			if(prevStickState->stickHeldHardUp) { tempInputs->bStickHardUp.released = true; inputStateChange[player] = true; }
			if(prevStickState->stickHeldHardDown) { tempInputs->bStickHardDown.released = true; inputStateChange[player] = true; }
			prevStickState->stickHeldHardUp = false;
			prevStickState->stickHeldHardDown = false;
		}
	}
}

void Main::JoyHat(Uint8 which,Uint8 hat,Uint8 value) 
{
	if(playerToSetUp > -1)
	{
		if(currentSettingMapping == SETTING_UP || currentSettingMapping == SETTING_DOWN ||
			currentSettingMapping == SETTING_LEFT || currentSettingMapping == SETTING_RIGHT)
		{
			ClearControls(playerToSetUp);
			mappings[playerToSetUp].hat = which;
			
			NextControl(true);
		}

		return;
	}

    InputStates * tempInputs = NULL;
	PreviousJoystickStates * prevStickState;
	int player = -1;
	
	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].hat == which)
		{
			tempInputs = curInputs[i];
			prevStickState = &prevJoystickStates[i];
			player = i;
		}
	}

	if(tempInputs == NULL)
	{
		return;
	}

	if(value & SDL_HAT_UP)
	{
		if(!(prevStickState->hat & SDL_HAT_UP))
		{
			tempInputs->bHatUp.pressed = true;
			inputStateChange[player] = true;
		}
	}
	else
	{
		if(prevStickState->hat & SDL_HAT_UP)
		{
			tempInputs->bHatUp.released = true;
			inputStateChange[player] = true;
		}
	}

	if(value & SDL_HAT_DOWN)
	{
		if(!(prevStickState->hat & SDL_HAT_DOWN))
		{
			tempInputs->bHatDown.pressed = true;
			inputStateChange[player] = true;
		}
	}
	else
	{
		if(prevStickState->hat & SDL_HAT_DOWN)
		{
			tempInputs->bHatDown.released = true;
			inputStateChange[player] = true;
		}
	}

	if(value & SDL_HAT_LEFT)
	{
		if(!(prevStickState->hat & SDL_HAT_LEFT))
		{
			tempInputs->bHatLeft.pressed = true;
			inputStateChange[player] = true;
		}
	}
	else
	{
		if(prevStickState->hat & SDL_HAT_LEFT)
		{
			tempInputs->bHatLeft.released = true;
			inputStateChange[player] = true;
		}
	}

	if(value & SDL_HAT_RIGHT)
	{
		if(!(prevStickState->hat & SDL_HAT_RIGHT))
		{
			tempInputs->bHatRight.pressed = true;
			inputStateChange[player] = true;
		}
	}
	else
	{
		if(prevStickState->hat & SDL_HAT_RIGHT)
		{
			tempInputs->bHatRight.released = true;
			inputStateChange[player] = true;
		}
	}

	prevStickState->hat = value;
}

void Main::JoyButtonDown(Uint8 which,Uint8 button) 
{
	if(playerToSetUp > -1)
	{
		switch(currentSettingMapping)
		{
			case SETTING_UP: mappings[playerToSetUp].buttonUp.joystick = which; mappings[playerToSetUp].buttonUp.button = button; break;
			case SETTING_DOWN: mappings[playerToSetUp].buttonDown.joystick = which; mappings[playerToSetUp].buttonDown.button = button; break;
			case SETTING_LEFT: mappings[playerToSetUp].buttonLeft.joystick = which; mappings[playerToSetUp].buttonLeft.button = button; break;
			case SETTING_RIGHT: mappings[playerToSetUp].buttonRight.joystick = which; mappings[playerToSetUp].buttonRight.button = button; break;
			case SETTING_JUMP: mappings[playerToSetUp].buttonJump.joystick = which; mappings[playerToSetUp].buttonJump.button = button; break;
			case SETTING_LIGHT: mappings[playerToSetUp].buttonLight.joystick = which; mappings[playerToSetUp].buttonLight.button = button; break;
			case SETTING_BLOCK: mappings[playerToSetUp].buttonBlock.joystick = which; mappings[playerToSetUp].buttonBlock.button = button; break;
		}

		NextControl(false);

		return;
	}

	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].buttonUp.joystick == which && mappings[i].buttonUp.button == button)
		{
			curInputs[i]->bButtonUp.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonDown.joystick == which && mappings[i].buttonDown.button == button)
		{
			curInputs[i]->bButtonDown.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonLeft.joystick == which && mappings[i].buttonLeft.button == button)
		{
			curInputs[i]->bButtonLeft.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonRight.joystick == which && mappings[i].buttonRight.button == button)
		{
			curInputs[i]->bButtonRight.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonJump.joystick == which && mappings[i].buttonJump.button == button)
		{
			curInputs[i]->bButtonJump.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonBlock.joystick == which && mappings[i].buttonBlock.button == button)
		{
			curInputs[i]->bButtonBlock.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonLight.joystick == which && mappings[i].buttonLight.button == button)
		{
			curInputs[i]->bButtonLight.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonHeavy.joystick == which && mappings[i].buttonHeavy.button == button)
		{
			curInputs[i]->bButtonHeavy.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonStart.joystick == which && mappings[i].buttonStart.button == button)
		{
			curInputs[i]->bButtonStart.pressed = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonSelect.joystick == which && mappings[i].buttonSelect.button == button)
		{
			curInputs[i]->bButtonSelect.pressed = true;
			inputStateChange[i] = true;
		}
	}
}

void Main::JoyButtonUp(Uint8 which,Uint8 button) 
{
	if(playerToSetUp > -1)
	{
		return;
	}

	for(int i = 0; i < MAX_PLAYERS; i++)
	{
		if(mappings[i].buttonUp.joystick == which && mappings[i].buttonUp.button == button)
		{
			curInputs[i]->bButtonUp.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonDown.joystick == which && mappings[i].buttonDown.button == button)
		{
			curInputs[i]->bButtonDown.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonLeft.joystick == which && mappings[i].buttonLeft.button == button)
		{
			curInputs[i]->bButtonLeft.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonRight.joystick == which && mappings[i].buttonRight.button == button)
		{
			curInputs[i]->bButtonRight.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonJump.joystick == which && mappings[i].buttonJump.button == button)
		{
			curInputs[i]->bButtonJump.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonBlock.joystick == which && mappings[i].buttonBlock.button == button)
		{
			curInputs[i]->bButtonBlock.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonLight.joystick == which && mappings[i].buttonLight.button == button)
		{
			curInputs[i]->bButtonLight.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonHeavy.joystick == which && mappings[i].buttonHeavy.button == button)
		{
			curInputs[i]->bButtonHeavy.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonStart.joystick == which && mappings[i].buttonStart.button == button)
		{
			curInputs[i]->bButtonStart.released = true;
			inputStateChange[i] = true;
		}
		else if(mappings[i].buttonSelect.joystick == which && mappings[i].buttonSelect.button == button)
		{
			curInputs[i]->bButtonSelect.released = true;
			inputStateChange[i] = true;
		}
	}
}

void Main::JoyBall(Uint8 which,Uint8 ball,Sint16 xrel,Sint16 yrel) 
{
    
}

void Main::Minimize() 
{
    
}

void Main::Restore() 
{
    
}

void Main::Resize(int w,int h) 
{
    
}

void Main::Expose() 
{
    
}

void Main::Exit()
{
    notDone = false;
}

int Main::ToggleFullScreen()
{
	if(fullScreen) { return SetFullScreen(false); }
	else { return SetFullScreen(true); }
}

int Main::SetFullScreen(bool newFullScreen)
{
	fullScreen = newFullScreen;

	if(int error = SetBestGameResolution() != 0)
	{
		return error;
	}

	UpdateLog("Resolution set.", false);

	int options = SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_OPENGL;

	if(!fullScreen)
	{
		screenResolutionX = gameResolutionX;
		screenResolutionY = gameResolutionY;
	}
	else
	{
		options = options | SDL_FULLSCREEN;
	}

	if(texCoordBufferID != 0)
	{
		glDeleteBuffers(1, &texCoordBufferID);
		texCoordBufferID = 0;
	}

	if(textureRegistry.size() > 0)
	{
		list<HSTexture*>::iterator trIt;
		for ( trIt=textureRegistry.begin(); trIt != textureRegistry.end(); trIt++)
		{
			glDeleteTextures(1, &(*trIt)->textureID);
			glDeleteBuffers(1, &(*trIt)->bufferID);

			(*trIt)->textureID = 0;
			(*trIt)->bufferID = 0;
		}
	}

	if((surf_display = SDL_SetVideoMode(screenResolutionX, screenResolutionY, 32, options)) == NULL)
	{
		UpdateLog("Error setting SDL video mode.", true);
		return -1;
	}

	UpdateLog("SDL video mode set.", false);

	glClearColor(0, 0, 0, 0);
	glClearDepth(1.0f);

	glViewport((screenResolutionX - gameResolutionX) / 2, (screenResolutionY - gameResolutionY) / 2, gameResolutionX, gameResolutionY);
 
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
 
	glOrtho(0, MAX_GAME_RESOLUTION_X, MAX_GAME_RESOLUTION_Y, 0, 1, -1);
 
	glMatrixMode(GL_MODELVIEW);
 
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
 
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	SDL_GL_SwapBuffers();

	//get the current opengl version
	int glMajorVersion = 0;
	int glMinorVersion = 0;
	glGetIntegerv(GL_MAJOR_VERSION, &glMajorVersion);
	glGetIntegerv(GL_MINOR_VERSION, &glMinorVersion);

	//gl major/minor version doesn't work, so use an older method
	if(glMajorVersion == 0)
	{
		string version; version.assign((const char *)glGetString(GL_VERSION));

		if(version.length() >= 1)
		{
			string major = version.substr(0, 1);
			glMajorVersion = atoi(major.c_str());
		}

		if(version.length() >= 3)
		{
			string minor = version.substr(2, 1);
			glMinorVersion = atoi(minor.c_str());
		}
	}

	stringstream sstm;
	sstm << "This machine is running OpenGL version " << glMajorVersion << "." << glMinorVersion << ".";
	UpdateLog(sstm.str(), false);

	if(glMajorVersion < 2)
	{
		UpdateLog("OpenGL version is older than 2.0.", true);
		return -1;
	}

	setupExtensions();
	if(shading_enabled == false || buffer_objects_enabled == false)
	{
		UpdateLog("OpenGL shaders and/or buffer objects are not enabled.", true);
		return -1; //just bail for now because I'm not handling this at the moment
	}

	UpdateLog("OpenGL shaders and buffer objects enabled.", false);

	//set up all the shader stuff
	shader_vert = 0;
	shader_frag = 0;
	shader_prog = 0;

	shader_prog = glCreateProgramObjectARB();
	shader_vert = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
	shader_frag = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);

	string shader_vert_source_string = loadSource("vertexShader1.glsl.txt");

	if(shader_vert_source_string.compare("READ ERROR") == 0)
	{
		UpdateLog("Could not read shader source file: vertexShader1.glsl.txt", true);
		return -1;
	}

	string shader_frag_source_string = loadSource("fragmentShader1.glsl.txt");

	if(shader_vert_source_string.compare("READ ERROR") == 0)
	{
		UpdateLog("Could not read shader source file: fragmentShader1.glsl.txt", true);
		return -1;
	}

	//add some newlines so it compiles correctly
	int verPos = shader_vert_source_string.find("#version 110", 0);
	shader_vert_source_string.insert(verPos + 12, "\n");
	verPos = shader_frag_source_string.find("#version 110", 0);
	shader_frag_source_string.insert(verPos + 12, "\n");
	int extPos = shader_frag_source_string.find("#extension GL_EXT_gpu_shader4 : enable", 0);
	shader_frag_source_string.insert(extPos + 38, "\n");

	const char * shader_vert_source = shader_vert_source_string.data();
	const char * shader_frag_source = shader_frag_source_string.data();

	int len = 0;
	len = shader_vert_source_string.length();
	glShaderSourceARB(shader_vert, 1, (const GLcharARB**)&shader_vert_source, &len);
	len = shader_frag_source_string.length();
	glShaderSourceARB(shader_frag, 1, (const GLcharARB**)&shader_frag_source, &len);

	string shaderError = "";
	shaderError = compileShader(shader_vert);
	if(shaderError.length() > 0 && shaderError.compare("Vertex shader was successfully compiled to run on hardware.") != 0)
	{
		UpdateLog("Error compiling vertex shader: " + shaderError, true);
		return -1;
	}

	shaderError = compileShader(shader_frag);
	if(shaderError.length() > 0 && shaderError.compare("Fragment shader was successfully compiled to run on hardware.") != 0)
	{
		UpdateLog("Error compiling fragment shader: " + shaderError, true);
		return -1;
	}

	glAttachObjectARB(shader_prog, shader_vert);
	glAttachObjectARB(shader_prog, shader_frag);

	shaderError = linkProgram(shader_prog);
	if(shaderError.length() > 0 && shaderError.compare("Fragment shader(s) linked, vertex shader(s) linked.") != 0)
	{
		UpdateLog("Error linking shader: " + shaderError, true);
		return -1;
	}

	glUseProgramObjectARB(shader_prog);

	UpdateLog("Shaders compiled and linked.", false);

	//set up some VBO stuff
	float * texCoord = new float[8];
	texCoord[0] = 0.0f;	texCoord[1] = 1.0f;
	texCoord[2] = 1.0f;	texCoord[3] = 1.0f;
	texCoord[4] = 1.0f;	texCoord[5] = 0.0f;
	texCoord[6] = 0.0f;	texCoord[7] = 0.0f;

	glGenBuffers(1, &texCoordBufferID);
	glBindBuffer(GL_ARRAY_BUFFER, texCoordBufferID);
	glBufferData(GL_ARRAY_BUFFER, 8*sizeof(float), texCoord, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	delete[] texCoord;

	if(textureRegistry.size() > 0)
	{
		//need to reload all textures
		list<HSTexture*>::iterator trIt;
		for ( trIt=textureRegistry.begin(); trIt != textureRegistry.end(); trIt++)
		{
			if(int error = LoadTGAToTexture((*trIt)) != 0)
			{
				return error;
			}
		}
	}

	return 0;
}

void Main::User(Uint8 type, int code, void* data1, void* data2) 
{
    string blah = "";
}

//main
int main( int argc, char* argv[] )
{
	HSLog = "";

	Main * theGame = new Main();
	int error = theGame->Execute();
	
	//output the error code, if there was one
	string errorText = "Application exited with no errors.\n\n";
	if(error != 0)
	{
		errorText = "An error occured while running the application. See log for details.\n\n";
	}

	errorText = errorText + "Log:\n" + HSLog;

	ofstream file;
	file.open("log.txt");
	file << errorText;
	file.close();

	return error;
}