#include "fighter.h"

///////////////
//FighterHold//
///////////////

FighterHold::FighterHold() : PhysicsObjectHold()
{
}

FighterHold::~FighterHold()
{
}

int FighterHold::Define(XMLElement * definition, string defFileDirectory, list<HSTexture*> * textureRegistry, list<HSAudio*> * audioRegistry, SDL_AudioSpec * obtainedAudioSpec)
{
	if(int error = PhysicsObjectHold::Define(definition, defFileDirectory, textureRegistry, audioRegistry, obtainedAudioSpec) != 0)
	{
		return error;
	}

	return 0;
}

///////////
//Fighter//
///////////
Fighter::Fighter() : PhysicsObject()
{
	firstPreviousTerrainBox = NULL;
	lastPreviousTerrainBox = NULL;

	terrainBoxesChanged = false;

	firstUprightTerrainBox = NULL;
	lastUprightTerrainBox = NULL;
	firstCrouchingTerrainBox = NULL;
	lastCrouchingTerrainBox = NULL;
	firstProneTerrainBox = NULL;
	lastProneTerrainBox = NULL;
	firstCompactTerrainBox = NULL;
	lastCompactTerrainBox = NULL;

	falls = false;
	state = JUMPING;
	facing = RIGHT;
	blocking = false;
	hitstunBlockability = MID;
	jump = NO_JUMP;
	airDash = NO_AIR_DASH;
	currentSurfaceFriction = 0;
	jumpStartup = false;
	turning = false;
	timeSinceWalkStop = 0;
	timeSinceRunStop = 0;
	runStopping = false;
	turning = false;
	sliding = false;
	attacking = false;
	lightNeutralAttack = false;
	jumpCancellable = false;
	walkAfterTurning = false;
	runAfterTurning = false;
	airVelocityCategory = LOW_VELOCITY;
	fastFall = false;
	airControl = NO_AIR_CONTROL;
	landingAction = NO_LANDING_ACTION;
	turnUponLanding = false;

	curHitstun = 0;
	curBlockstun = 0;

	fighterEventHolds.standing = NULL;
	fighterEventHolds.turn = NULL;
	fighterEventHolds.walk = NULL;
	fighterEventHolds.walking = NULL;
	fighterEventHolds.walkingTurn = NULL;
	fighterEventHolds.run = NULL;
	fighterEventHolds.running = NULL;
	fighterEventHolds.runningTurn = NULL;
	fighterEventHolds.runningStop = NULL;
	fighterEventHolds.crouch = NULL;
	fighterEventHolds.crouching = NULL;
	fighterEventHolds.crouchingTurn = NULL;
	fighterEventHolds.stand = NULL;

	fighterEventHolds.jumpNeutralStart = NULL;
	fighterEventHolds.jumpNeutralStartAir = NULL;
	fighterEventHolds.jumpNeutralRising = NULL;
	fighterEventHolds.jumpNeutralFall = NULL;
	fighterEventHolds.jumpNeutralFalling = NULL;
	fighterEventHolds.jumpNeutralLand = NULL;
	fighterEventHolds.jumpBackwardRising = NULL;
	fighterEventHolds.jumpBackwardFall = NULL;

	fighterEventHolds.airDashForward = NULL;
	fighterEventHolds.airDashBackward = NULL;
        
	fighterEventHolds.blockHigh = NULL;
	fighterEventHolds.blockLow = NULL;
	fighterEventHolds.blockAir = NULL;

	fighterEventHolds.hitstunLightHighStanding = NULL;
	fighterEventHolds.hitstunLightMidStanding = NULL;
	fighterEventHolds.hitstunLightLowStanding = NULL;
	fighterEventHolds.hitstunLightMidCrouching = NULL;
	fighterEventHolds.hitstunLightAir = NULL;
    
	fighterEventHolds.attackLightNeutralGround = NULL;
	fighterEventHolds.attackLightDownGround = NULL;
	fighterEventHolds.attackLightUpGround = NULL;
	fighterEventHolds.attackLightForwardGround = NULL;
	fighterEventHolds.attackLightNeutralAir = NULL;
	fighterEventHolds.attackLightDownAir = NULL;
	fighterEventHolds.attackLightUpAir = NULL;
	fighterEventHolds.attackLightForwardAir = NULL;
	fighterEventHolds.attackLightBackwardAir = NULL;

	walkSpeed = 0;
	runSpeed = 0;
	jumpSpeed = 0;
	forwardAirDashSpeed = 0;
	forwardAirDashDuration = 0;
	backwardAirDashSpeed = 0;
	backwardAirDashDuration = 0;
	curAirDashDuration = 0;
	stepHeight = 0;
	airActions = 1;
	curAirActions = airActions;
	airControlAccel = 0;
	maxAirControlSpeed = 0;

	hud = NULL;
}

Fighter::~Fighter()
{
}

int Fighter::Define(XMLElement * definition, string defFileDirectory, list<HSTexture*> * textureRegistry, list<HSPalette*> * paletteRegistry, list<HSAudio*> * audioRegistry, SDL_AudioSpec * obtainedAudioSpec)
{
	if(int error = PhysicsObject::Define(definition, defFileDirectory, textureRegistry, paletteRegistry, audioRegistry, obtainedAudioSpec) != 0)
	{
		return error;
	}

	//get the upright terrain boxes
	XMLElement * uprightTerrainBox;;
	if((uprightTerrainBox = definition->FirstChildElement("UprightTerrainBox")) != NULL)
	{
		HSBox * newUprightTerrainBox = new HSBox();
		newUprightTerrainBox->nextBox = NULL;
					
		//add the terrain box to the object
		if(int error = AddUprightTerrainBox(newUprightTerrainBox, uprightTerrainBox) != 0)
		{
			return error; //there was an error adding the terrain box
		}
	}

	//get the crouching terrain boxes
	XMLElement * crouchingTerrainBox;
	if((crouchingTerrainBox = definition->FirstChildElement("CrouchingTerrainBox")) != NULL)
	{
		HSBox * newCrouchingTerrainBox = new HSBox();
		newCrouchingTerrainBox->nextBox = NULL;
					
		//add the terrain box to the object
		if(int error = AddCrouchingTerrainBox(newCrouchingTerrainBox, crouchingTerrainBox) != 0)
		{
			return error; //there was an error adding the terrain box
		}
	}

	//get the prone terrain boxes
	XMLElement * proneTerrainBox;
	if((proneTerrainBox = definition->FirstChildElement("ProneTerrainBox")) != NULL)
	{
		HSBox * newProneTerrainBox = new HSBox();
		newProneTerrainBox->nextBox = NULL;
					
		//add the terrain box to the object
		if(int error = AddProneTerrainBox(newProneTerrainBox, proneTerrainBox) != 0)
		{
			return error; //there was an error adding the terrain box
		}
	}

	//get the compact terrain boxes
	XMLElement * compactTerrainBox;
	if((compactTerrainBox = definition->FirstChildElement("CompactTerrainBox")) != NULL)
	{
		HSBox * newCompactTerrainBox = new HSBox();
		newCompactTerrainBox->nextBox = NULL;
					
		//add the terrain box to the object
		if(int error = AddCompactTerrainBox(newCompactTerrainBox, compactTerrainBox) != 0)
		{
			return error; //there was an error adding the terrain box
		}
	}

	ChangeTerrainBoxes(firstUprightTerrainBox);

	//get the walkSpeed
	definition->QueryFloatAttribute("walkSpeed", &walkSpeed);

	//get the runSpeed
	definition->QueryFloatAttribute("runSpeed", &runSpeed);

	//get the jumpSpeed
	definition->QueryFloatAttribute("jumpSpeed", &jumpSpeed);

	//get the stepHeight
	definition->QueryFloatAttribute("stepHeight", &stepHeight);

	//get air dash statistics
	definition->QueryFloatAttribute("forwardAirDashSpeed", &forwardAirDashSpeed);
	definition->QueryFloatAttribute("backwardAirDashSpeed", &backwardAirDashSpeed);
	definition->QueryIntAttribute("forwardAirDashDuration", &forwardAirDashDuration);
	definition->QueryIntAttribute("backwardAirDashDuration", &backwardAirDashDuration);

	//get air control statistics
	definition->QueryFloatAttribute("airControlAccel", &airControlAccel);
	definition->QueryFloatAttribute("maxAirControlSpeed", &maxAirControlSpeed);

	return 0;
}

void Fighter::ChangeTerrainBoxes(HSBox * newFirstTerrainBox)
{
	//don't bother if the new box is same as the current one
	if(newFirstTerrainBox == firstTerrainBox) { return; }

	//change the box, saving the current one
	firstPreviousTerrainBox = firstTerrainBox;
	firstTerrainBox = newFirstTerrainBox;

	terrainBoxesChanged = true;
}

int Fighter::AdvanceHolds()
{
	//update turn timers
	if(timeSinceWalkStop < MOVING_TURN_THRESHOLD) { timeSinceWalkStop++; }
	if(timeSinceRunStop < MOVING_TURN_THRESHOLD) { timeSinceRunStop++; }

	//update air dash timer
	if(state == AIR_DASHING)
	{
		falls = false;
		curAirDashDuration++;
		if(airDash == FORWARD_AIR_DASH && curAirDashDuration >= forwardAirDashDuration)
		{
			falls = true;
			state = JUMPING;
			ChangeHold(fighterEventHolds.jumpNeutralFall);
			curAirDashDuration = 0;
			airDash = NO_AIR_DASH;
		}
		else if(airDash == BACKWARD_AIR_DASH && curAirDashDuration >= backwardAirDashDuration)
		{
			falls = true;
			state = JUMPING;
			ChangeHold(fighterEventHolds.jumpBackwardFall);
			curAirDashDuration = 0;
			airDash = NO_AIR_DASH;
		}
	}

	//update stun
	bool hitstunEnd = false;
	bool blockstunEnd = false;
	if(curHitstun > 0)
	{
		curHitstun--;
		blocking = false;
		if(curHitstun <= 0)
		{
			hitstunEnd = true;
		}
	}
	if(curBlockstun > 0)
	{
		curBlockstun--;
		blocking = false;
		if(curBlockstun > 0)
		{
			blocking = true;
		}
		else
		{
			blockstunEnd = true;
		}
	}

	//do the base hold advancement
	if(int error = HSObject::AdvanceHolds() != 0) { return error; }

	//if stun ended, apply the new appropriate hold here
	if(hitstunEnd || blockstunEnd)
	{
		ChangeHold(GetDefaultHold());
	}

	return 0;
}

int Fighter::Event(InputStates * inputHistory, int frame)
{
	landingAction = NO_LANDING_ACTION;
	turnUponLanding = false;

	//stun
	if(curHitstun > 0)
	{
		return 0;
	}

	//left/right movement
	if(inputHistory->bKeyLeft.held || inputHistory->bButtonLeft.held || inputHistory->bHatLeft.held || inputHistory->bStickLeft.held)
	{
		//air control
		if(state == JUMPING)
		{
			airControl = CONTROL_LEFT;
			landingAction = MOVE;
			if(facing == RIGHT)
			{
				turnUponLanding = true;
			}
		}

		if(facing == LEFT)
		{
			if(state == STANDING && (turning || runStopping) && (ForwardWasTapped(inputHistory, frame, RUN_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
			{
				runAfterTurning = true;
			}
			else if(!attacking && !blocking && state == STANDING && !turning && !runStopping)
			{
				if(runAfterTurning || ForwardWasTapped(inputHistory, frame, RUN_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD))
				{
					ChangeHold(fighterEventHolds.run);
					if(runAfterTurning)
					{
						walkAfterTurning = false;
						runAfterTurning = false;
					}
					state = RUNNING;
				}
				else
				{
					if(walkAfterTurning)
					{
						ChangeHold(fighterEventHolds.walking);
						walkAfterTurning = false;
						runAfterTurning = false;
					}
					else
					{
						ChangeHold(fighterEventHolds.walk);
					}
					state = WALKING;
				}
			}
			else if(!attacking && !blocking && state == JUMPING)
			{
				if(curAirActions > 0 && (ForwardWasTapped(inputHistory, frame, AIR_DASH_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
				{
					ChangeHold(fighterEventHolds.airDashForward);
					state = AIR_DASHING;
					airDash = FORWARD_AIR_DASH;
					airControl = NO_AIR_CONTROL;
					curAirActions--;
					landingAction = DASH;
					turnUponLanding = false;
				}
			}
		}
		else
		{
			if(blocking && state != JUMPING)
			{
				facing = LEFT;
			}
			else if(!attacking && !blocking && ((state == STANDING && timeSinceWalkStop < MOVING_TURN_THRESHOLD) || state == WALKING))
			{
				ChangeHold(fighterEventHolds.walkingTurn);
				facing = LEFT;
				turning = true;
				walkAfterTurning = false;
				runAfterTurning = false;
				if(ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)) { runAfterTurning = true; }
				else { walkAfterTurning = true; }
				timeSinceWalkStop = MOVING_TURN_THRESHOLD + 1;
				state = STANDING;
			}
			else if(!attacking && !blocking && state == STANDING && !runStopping)
			{
				ChangeHold(fighterEventHolds.turn);
				facing = LEFT;
				turning = true;
				walkAfterTurning = false;
				runAfterTurning = false;
				if(ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)) { runAfterTurning = true; }
				state = STANDING;
			}
			else if(!attacking && !blocking && state == CROUCHING)
			{
				ChangeHold(fighterEventHolds.crouchingTurn);
				facing = LEFT;
				turning = true;
			}
			else if(!attacking && !blocking && ((state == STANDING && timeSinceRunStop < MOVING_TURN_THRESHOLD && runStopping) || state == RUNNING))
			{
				ChangeHold(fighterEventHolds.runningTurn);
				facing = LEFT;
				turning = true;
				runStopping = false;
				walkAfterTurning = false;
				runAfterTurning = true;
				timeSinceRunStop = MOVING_TURN_THRESHOLD + 1;
				sliding = true;
				state = STANDING;
			}
			else if(!attacking && !blocking && state == JUMPING)
			{
				if(curAirActions > 0 && (BackwardWasTapped(inputHistory, frame, AIR_DASH_INPUT_FRAMES) || BackwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
				{
					ChangeHold(fighterEventHolds.airDashBackward);
					state = AIR_DASHING;
					airDash = BACKWARD_AIR_DASH;
					curAirActions--;
					landingAction = DASH;
					turnUponLanding = true;
				}
			}
		}
	}
	else if(inputHistory->bKeyRight.held || inputHistory->bButtonRight.held || inputHistory->bHatRight.held || inputHistory->bStickRight.held)
	{
		//air control
		if(state == JUMPING)
		{
			airControl = CONTROL_RIGHT;
			landingAction = MOVE;
			if(facing == LEFT)
			{
				turnUponLanding = true;
			}
		}

		if(facing == LEFT)
		{
			if(blocking && state != JUMPING)
			{
				facing = RIGHT;
			}
			else if(!attacking && !blocking && ((state == STANDING && timeSinceWalkStop < MOVING_TURN_THRESHOLD) || state == WALKING))
			{
				ChangeHold(fighterEventHolds.walkingTurn);
				facing = RIGHT;
				turning = true;
				walkAfterTurning = false;
				runAfterTurning = false;
				if(ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)) { runAfterTurning = true; }
				else { walkAfterTurning = true; }
				timeSinceWalkStop = MOVING_TURN_THRESHOLD + 1;
				state = STANDING;
			}
			else if(!attacking && !blocking && state == STANDING && !runStopping)
			{
				ChangeHold(fighterEventHolds.turn);
				facing = RIGHT;
				turning = true;
				walkAfterTurning = false;
				runAfterTurning = false;
				if(ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)) { runAfterTurning = true; }
				state = STANDING;
			}
			else if(!attacking && !blocking && state == CROUCHING)
			{
				ChangeHold(fighterEventHolds.crouchingTurn);
				facing = RIGHT;
				turning = true;
			}
			else if(!attacking && !blocking && ((state == STANDING && timeSinceRunStop < MOVING_TURN_THRESHOLD && runStopping) || state == RUNNING))
			{
				ChangeHold(fighterEventHolds.runningTurn);
				facing = RIGHT;
				turning = true;
				runStopping = false;
				walkAfterTurning = false;
				runAfterTurning = true;
				sliding = true;
				timeSinceRunStop = MOVING_TURN_THRESHOLD + 1;
				state = STANDING;
			}
			else if(!attacking && !blocking && state == JUMPING)
			{
				if(curAirActions > 0 && (BackwardWasTapped(inputHistory, frame, AIR_DASH_INPUT_FRAMES) || BackwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
				{
					ChangeHold(fighterEventHolds.airDashBackward);
					state = AIR_DASHING;
					airDash = BACKWARD_AIR_DASH;
					curAirActions--;
					landingAction = DASH;
					turnUponLanding = true;
				}
			}
		}
		else
		{
			if(state == STANDING && (turning || runStopping) && (ForwardWasTapped(inputHistory, frame, RUN_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
			{
				runAfterTurning = true;
			}
			else if(!attacking && !blocking && state == STANDING && !turning && !runStopping)
			{
				if(runAfterTurning || ForwardWasTapped(inputHistory, frame, RUN_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD))
				{ 
					ChangeHold(fighterEventHolds.run);
					if(runAfterTurning)
					{
						walkAfterTurning = false;
						runAfterTurning = false;
					}
					state = RUNNING;
				}
				else
				{
					if(runAfterTurning)
					{
						ChangeHold(fighterEventHolds.walking);
						walkAfterTurning = false;
						runAfterTurning = false;
					}
					else
					{
						ChangeHold(fighterEventHolds.walk);
					}
					state = WALKING;
				}
			}
			else if(!attacking && !blocking && state == JUMPING)
			{
				if(curAirActions > 0 && (ForwardWasTapped(inputHistory, frame, AIR_DASH_INPUT_FRAMES) || ForwardHardPressed(inputHistory, frame, HARD_PRESS_THRESHOLD)))
				{
					ChangeHold(fighterEventHolds.airDashForward);
					state = AIR_DASHING;
					airDash = FORWARD_AIR_DASH;
					airControl = NO_AIR_CONTROL;
					curAirActions--;
					landingAction = DASH;
					turnUponLanding = false;
				}
			}
		}
	}
	else
	{
		walkAfterTurning = false;
		runAfterTurning = false;
		if(state == WALKING)
		{
			ChangeHold(fighterEventHolds.standing);
			timeSinceWalkStop = 0;
			state = STANDING;
		}
		else if(state == RUNNING)
		{
			ChangeHold(fighterEventHolds.runningStop);
			timeSinceRunStop = 0;
			runStopping = true;
			sliding = true;
			state = STANDING;
		}
	}

	//crouch
	if(inputHistory->bKeyDown.held || inputHistory->bButtonDown.held || inputHistory->bHatDown.held || inputHistory->bStickDown.held)
	{
		if(!attacking && (state == STANDING || state == WALKING || state == RUNNING))
		{
			if(blocking || curBlockstun > 0)
			{
				ChangeHold(fighterEventHolds.blockLow);
			}
			else
			{
				ChangeHold(fighterEventHolds.crouch);
			}
			walkAfterTurning = false;
			runAfterTurning = false;
			turning = false;
			runStopping = false;
			if(state == RUNNING) { sliding = true; }
			state = CROUCHING;
		}
		else if(state == JUMPING)
		{
			landingAction = CROUCH;
			if(inputHistory->bKeyJump.held || inputHistory->bButtonJump.held)
			{
				ignoreJumpThroughTerrain = true;
			}
		}
	}
	else if(!attacking && state == CROUCHING)
	{
		if(blocking || curBlockstun > 0)
		{
			ChangeHold(fighterEventHolds.blockHigh);
		}
		else
		{
			ChangeHold(fighterEventHolds.stand);
		}
		turning = false;
		state = STANDING;
	}

	//jump
	if(inputHistory->frame == frame && (!attacking || (jumpCancellable && victims.size() > 0)) && !jumpStartup && !blocking && curBlockstun <= 0 && (inputHistory->bKeyJump.pressed || inputHistory->bButtonJump.pressed))
	{
		jumpCancellable = false;
		if(state != JUMPING && state != AIRBORN && state != AIR_DASHING)
		{
			//ground jump
			if(state == CROUCHING)
			{
				ignoreJumpThroughTerrain = true;
			}
			else
			{
				ChangeHold(fighterEventHolds.jumpNeutralStart);
				if(state == STANDING)
				{
					if(turning || runStopping || attacking)
					{
						turning = false;
						runStopping = false;
						attacking = false;
						//base it on which direction is being held
						if(inputHistory->bKeyLeft.held || inputHistory->bButtonLeft.held || inputHistory->bHatLeft.held || inputHistory->bStickLeft.held)
						{
							if(facing == LEFT)
							{
								jump = WALKING_JUMP;
							}
							else
							{
								jump = BACKWARD_JUMP;
							}
						}
						else if(inputHistory->bKeyRight.held || inputHistory->bButtonRight.held || inputHistory->bHatRight.held || inputHistory->bStickRight.held)
						{
							if(facing == LEFT)
							{
								jump = BACKWARD_JUMP;
							}
							else
							{
								jump = WALKING_JUMP;
							}
						}
						else
						{
							jump = STANDING_JUMP;
						}
					}
					else
					{
						jump = STANDING_JUMP;
					}
				}
				if(state == WALKING)
				{
					jump = WALKING_JUMP;
				}
				if(state == RUNNING)
				{
					jump = RUNNING_JUMP;
				}
				walkAfterTurning = false;
				runAfterTurning = false;
				jumpStartup = true;
				state = JUMPING;
			}
		}
		else if(state == JUMPING && !(inputHistory->bKeyDown.held || inputHistory->bButtonDown.held || inputHistory->bHatDown.held || inputHistory->bStickDown.held))
		{
			//air jump
			if(curAirActions > 0)
			{
				ChangeHold(fighterEventHolds.jumpNeutralStartAir);
				if(inputHistory->bKeyLeft.held || inputHistory->bButtonLeft.held || inputHistory->bHatLeft.held || inputHistory->bStickLeft.held)
				{
					if(facing == LEFT)
					{
						if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
						{
							jump = RUNNING_JUMP;
						}
						else
						{
							jump = WALKING_JUMP;
						}
					}
					else
					{
						if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
						{
							jump = STANDING_JUMP;
						}
						else
						{
							jump = BACKWARD_JUMP;
						}
					}
				}
				else if(inputHistory->bKeyRight.held || inputHistory->bButtonRight.held || inputHistory->bHatRight.held || inputHistory->bStickRight.held)
				{
					if(facing == LEFT)
					{
						if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
						{
							jump = STANDING_JUMP;
						}
						else
						{
							jump = BACKWARD_JUMP;
						}
					}
					else
					{
						if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
						{
							jump = RUNNING_JUMP;
						}
						else
						{
							jump = WALKING_JUMP;
						}
					}
				}
				else
				{
					if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
					{
						jump = WALKING_JUMP;
					}
					else
					{
						jump = STANDING_JUMP;
					}
				}
				attacking = false;
				curAirActions--;
			}
		}
	}

	//blocking
	if(!attacking && !blocking && curBlockstun <= 0 && (inputHistory->bKeyBlock.held || inputHistory->bButtonBlock.held))
	{
		if(state == STANDING || state == WALKING || state == RUNNING || state == CROUCHING || (state == JUMPING && jumpStartup))
		{
			if(state == RUNNING)
			{
				sliding = true;
			}
			if(inputHistory->bKeyDown.held || inputHistory->bButtonDown.held || inputHistory->bStickDown.held || inputHistory->bHatDown.held)
			{
				ChangeHold(fighterEventHolds.blockLow);
				state = CROUCHING;
			}
			else
			{
				ChangeHold(fighterEventHolds.blockHigh);
				state = STANDING;
			}
			walkAfterTurning = false;
			runAfterTurning = false;
			turning = false;
			runStopping = false;
			jump = NO_JUMP;
			blocking = true;
			jumpStartup = false;
		}
		else if(state == JUMPING)
		{
			ChangeHold(fighterEventHolds.blockAir);
			blocking = true;
		}
	}
	else if(blocking && curBlockstun <= 0 && !inputHistory->bKeyBlock.held && !inputHistory->bButtonBlock.held)
	{
		if(state == STANDING)
		{
			ChangeHold(fighterEventHolds.standing);
		}
		else if(state == CROUCHING)
		{
			ChangeHold(fighterEventHolds.crouching);
		}
		else if(state == JUMPING)
		{
			if(vel.y < 0)
			{
				if((facing == LEFT && vel.x > 0) || (facing == RIGHT && vel.x < 0))
				{
					ChangeHold(fighterEventHolds.jumpBackwardRising);
				}
				else
				{
					ChangeHold(fighterEventHolds.jumpNeutralRising);
				}
			}
			else
			{
				ChangeHold(fighterEventHolds.jumpNeutralFalling);
			}
		}
		turning = false;
		runStopping = false;
		blocking = false;
	}

	if((attacking && (!lightNeutralAttack || victims.size() <= 0)) || curBlockstun > 0) { return 0; }

	//light attacks
	if(inputHistory->frame == frame && (inputHistory->bKeyLight.pressed || inputHistory->bButtonLight.pressed))
	{
		if(state == STANDING || state == CROUCHING || state == WALKING || state == RUNNING || (state == JUMPING && jumpStartup))
		{
			lightNeutralAttack = false;
			jumpCancellable = false;
			if(fighterEventHolds.attackLightUpGround != NULL && (inputHistory->bKeyUp.held || inputHistory->bButtonUp.held || inputHistory->bHatUp.held || inputHistory->bStickUp.held))
			{
				ChangeHold(fighterEventHolds.attackLightUpGround);
				GroundAttackExecuted();
				jumpCancellable = true;
			}
			else if(fighterEventHolds.attackLightDownGround != NULL && (inputHistory->bKeyDown.held || inputHistory->bButtonDown.held || inputHistory->bHatDown.held || inputHistory->bStickDown.held))
			{
				ChangeHold(fighterEventHolds.attackLightDownGround);
				GroundAttackExecuted();
				lightNeutralAttack = true;
				state = CROUCHING;
			}
			else if(fighterEventHolds.attackLightForwardGround != NULL && ForwardHeld(inputHistory))
			{
				ChangeHold(fighterEventHolds.attackLightForwardGround);
				GroundAttackExecuted();
			}
			else if(fighterEventHolds.attackLightNeutralGround != NULL)
			{
				ChangeHold(fighterEventHolds.attackLightNeutralGround);
				GroundAttackExecuted();
				lightNeutralAttack = true;
				jumpCancellable = true;
			}
		}
		else if(state == JUMPING)
		{
			lightNeutralAttack = false;
			jumpCancellable = false;
			if(fighterEventHolds.attackLightUpAir != NULL && (inputHistory->bKeyUp.held || inputHistory->bButtonUp.held || inputHistory->bHatUp.held || inputHistory->bStickUp.held))
			{
				ChangeHold(fighterEventHolds.attackLightUpAir);
				AirAttackExecuted();
				jumpCancellable = true;
				landingAction = UP_LIGHT;
			}
			else if(fighterEventHolds.attackLightDownAir != NULL && (inputHistory->bKeyDown.held || inputHistory->bButtonDown.held || inputHistory->bHatDown.held || inputHistory->bStickDown.held))
			{
				ChangeHold(fighterEventHolds.attackLightDownAir);
				AirAttackExecuted();
				landingAction = DOWN_LIGHT;
			}
			else if(fighterEventHolds.attackLightForwardAir != NULL && (ForwardHeld(inputHistory)))
			{
				ChangeHold(fighterEventHolds.attackLightForwardAir);
				AirAttackExecuted();
				landingAction = FORWARD_LIGHT;
				turnUponLanding = false;
			}
			else if(fighterEventHolds.attackLightBackwardAir != NULL && (BackwardHeld(inputHistory)))
			{
				ChangeHold(fighterEventHolds.attackLightBackwardAir);
				AirAttackExecuted();
				landingAction = FORWARD_LIGHT;
				turnUponLanding = true;
			}
			else if(fighterEventHolds.attackLightNeutralAir != NULL)
			{
				ChangeHold(fighterEventHolds.attackLightNeutralAir);
				AirAttackExecuted();
				lightNeutralAttack = true;
				jumpCancellable = true;
				landingAction = NEUTRAL_LIGHT;
			}
		}
	}

	return 0;
}

int Fighter::Update()
{
	if(int error = PhysicsObject::Update() != 0) { return error; } //there was an error in the base update

	//handle facing direction
	if(facing == LEFT)
	{
		hFlip = true;
	}
	else
	{
		hFlip = false;
	}

	//handle terrain boxes
	if(state == STANDING || state == WALKING || state == RUNNING || state == JUMPING)
	{
		ChangeTerrainBoxes(firstUprightTerrainBox);
	}
	else if(state == CROUCHING)
	{
		ChangeTerrainBoxes(firstUprightTerrainBox);
	}
	else if(state == AIR_DASHING || state == AIRBORN)
	{
		ChangeTerrainBoxes(firstUprightTerrainBox);
	}

	//handle motion
	if((state != JUMPING && state != AIRBORN) || (state == JUMPING && jumpStartup))
	{
		vel.y = 0;
	}

	if(state == STANDING || state == CROUCHING || (state == JUMPING && jumpStartup))
	{
		if(sliding)
		{
			float totalFriction = (1 - friction) * (1 - currentSurfaceFriction);
			vel.x *= totalFriction;
			if(abs(vel.x) < MIN_VEL)
			{
				vel.x = 0;
				sliding = false;
			}
		}
		else
		{
			vel.x = 0;
		}
	}
	else
	{
		sliding = false;
		walkAfterTurning = false;
		runAfterTurning = false;
	}
	if(jump == STANDING_JUMP && !jumpStartup)
	{
		vel.x = 0;
	}
	if(state == WALKING)
	{
		vel.x = walkSpeed * facing;
	}
	if(state == RUNNING)
	{
		vel.x = runSpeed * facing;
	}
	if(state == AIR_DASHING)
	{
		if(airDash == FORWARD_AIR_DASH)
		{
			falls = false;
			vel.x = forwardAirDashSpeed * facing;
		}
		else if(airDash == BACKWARD_AIR_DASH)
		{
			falls = false;
			vel.x = backwardAirDashSpeed * -facing;
		}
		if((facing == LEFT && vel.x < -maxAirControlSpeed) || (facing == RIGHT && vel.x > maxAirControlSpeed))
		{
			airVelocityCategory = HIGH_VELOCITY_FORWARD;
		}
		else if((facing == RIGHT && vel.x < -maxAirControlSpeed) || (facing == LEFT && vel.x > maxAirControlSpeed))
		{
			airVelocityCategory = HIGH_VELOCITY_BACKWARD;
		}
		else
		{
			airVelocityCategory = LOW_VELOCITY;
		}
	}
	if(state == JUMPING && !jumpStartup)
	{
		//air control
		if(airControl != NO_AIR_CONTROL)
		{
			if(airControl == CONTROL_LEFT &&  vel.x > -maxAirControlSpeed)
			{
				vel.x -= airControlAccel;
				if(vel.x < -maxAirControlSpeed)
				{
					vel.x = -maxAirControlSpeed;
				}
			}
			else if(airControl == CONTROL_RIGHT &&  vel.x < maxAirControlSpeed)
			{
				vel.x += airControlAccel;
				if(vel.x > maxAirControlSpeed)
				{
					vel.x = maxAirControlSpeed;
				}
			}
			airControl = NO_AIR_CONTROL;
		}

		//make jump happen
		if(jump != NO_JUMP)
		{
			if(curAirActions == airActions)
			{
				ChangeHold(fighterEventHolds.jumpNeutralRising);
			}

			if(jump == BACKWARD_JUMP)
			{
				vel.x = walkSpeed * -facing;
				if(curAirActions == airActions)
				{
					ChangeHold(fighterEventHolds.jumpBackwardRising);
				}
			}
			else if(jump == STANDING_JUMP)
			{
				vel.x = 0;
			}
			else if(jump == WALKING_JUMP)
			{
				vel.x = walkSpeed * facing;
			}
			else if(jump == RUNNING_JUMP)
			{
				vel.x = runSpeed * facing;
			}

			vel.y = -jumpSpeed;
			jump = NO_JUMP;
			falls = true;
		}

		//figure out the air velocity category
		if((facing == LEFT && vel.x < -maxAirControlSpeed) || (facing == RIGHT && vel.x > maxAirControlSpeed))
		{
			airVelocityCategory = HIGH_VELOCITY_FORWARD;
		}
		else if((facing == RIGHT && vel.x < -maxAirControlSpeed) || (facing == LEFT && vel.x > maxAirControlSpeed))
		{
			airVelocityCategory = HIGH_VELOCITY_BACKWARD;
		}
		else
		{
			airVelocityCategory = LOW_VELOCITY;
		}

		//jump animation changes based on change in velocity
		if(!attacking && !blocking && curHitstun <= 0 && curBlockstun <= 0)
		{
			if(vel.y < 0)
			{
				//jump rising
				if((facing == LEFT && prevVel.x <= 0 && vel.x > 0) || (facing == RIGHT && prevVel.x >= 0 && vel.x < 0))
				{
					ChangeHold(fighterEventHolds.jumpBackwardRising);
				}
				else if((facing == LEFT && prevVel.x > 0 && vel.x <= 0) || (facing == RIGHT && prevVel.x < 0 && vel.x >= 0))
				{
					ChangeHold(fighterEventHolds.jumpNeutralRising);
				}
			}
			if(prevVel.y < 0 && vel.y >= 0)
			{
				//jump peak
				if((facing == LEFT && vel.x > 0) || (facing == RIGHT && vel.x < 0))
				{
					ChangeHold(fighterEventHolds.jumpBackwardFall);
				}
				else
				{
					ChangeHold(fighterEventHolds.jumpNeutralFall);
				}
			}
		}
	}

	return 0;
}

void Fighter::HandleChangeInTerrainBoxes(list<HSObject*> * gameObjects)
{
	if(!terrainBoxesChanged) { return; }

	terrainBoxesChanged = false;

	return;

	//check if the new box's lower boundary is lower than the old one
	HSBox * oldBox = firstPreviousTerrainBox;
	float oldLowerBound = oldBox->offset.y + oldBox->height;
	oldBox = oldBox->nextBox;
	while(oldBox != NULL)
	{
		float lowerBound = oldBox->offset.y + oldBox->height;

		if(lowerBound > oldLowerBound) { oldLowerBound = lowerBound; }

		oldBox = oldBox->nextBox;
	}

	HSBox * newBox = firstTerrainBox;
	bool newIsLower = false;
	while(newBox != NULL)
	{
		float lowerBound = newBox->offset.y + newBox->height;

		if(lowerBound > oldLowerBound)
		{
			newIsLower = true;
			break;
		}
	}

	//new box's lower bound isn't lower than the old one, so change boxes and return
	if(!newIsLower) { return; }

	bool adjustmentRequired = true;
	HSBox * selfTerrainBox;
	TerrainObject * targetTerrainObject;
	HSBox * targetTerrainBox;

	while(adjustmentRequired)
	{
		adjustmentRequired = false;

		//check against jump-through platforms
		list<HSObject*>::iterator objIt;
		for ( objIt=gameObjects->begin(); objIt != gameObjects->end(); objIt++)
		{
			bool collisionWithOldTerrainBoxes = false;

			//loop through this object's previous terrain boxes
			selfTerrainBox = firstPreviousTerrainBox;
			while(selfTerrainBox != NULL)
			{
				//loop through the target's terrain boxes
				if(!(*objIt)->IsTerrain()) { continue; }
				targetTerrainObject = (TerrainObject*)(*objIt);

				targetTerrainBox = targetTerrainObject->firstTerrainBox;
				while(targetTerrainBox != NULL)
				{
					//check for collisions
					if(AreRectanglesColliding(&pos, selfTerrainBox, &(*objIt)->pos, targetTerrainBox))
					{
						collisionWithOldTerrainBoxes = true;
						break;
					}

					targetTerrainBox = targetTerrainBox->nextBox;
				}
				if(collisionWithOldTerrainBoxes)
				{
					break;
				}
				selfTerrainBox = selfTerrainBox->nextBox;
			}

			if(collisionWithOldTerrainBoxes)
			{
				continue;
			}

			//loop through this object's current terrain boxes
			selfTerrainBox = firstTerrainBox;
			while(selfTerrainBox != NULL)
			{
				//loop through the target's terrain boxes
				if(!(*objIt)->IsTerrain()) { continue; }
				targetTerrainObject = (TerrainObject*)(*objIt);

				targetTerrainBox = targetTerrainObject->firstTerrainBox;
				while(targetTerrainBox != NULL)
				{
					//check for collisions
					if(AreRectanglesColliding(&pos, selfTerrainBox, &(*objIt)->pos, targetTerrainBox))
					{
						adjustmentRequired = true;
						break;
					}

					targetTerrainBox = targetTerrainBox->nextBox;
				}
				if(adjustmentRequired)
				{
					break;
				}
				selfTerrainBox = selfTerrainBox->nextBox;
			}

			if(adjustmentRequired) { break; }
		}

		if(!adjustmentRequired) { break; }

		//move this object upward to put the selfTerrainBox just above the targetTerrainBox
		pos.y = targetTerrainObject->pos.y + targetTerrainBox->offset.y + selfTerrainBox->height - pos.y;
	}
}

int Fighter::HandleTerrainCollision(list<HSObject*> * gameObjects, TerrainCollisionResult * result, HSVect2D * ownPos, HSVect2D * ownPrevPos)
{
	HSVect2D handlePos;
	handlePos.x = pos.x;
	handlePos.y = pos.y;
	HSVect2D handlePrevPos;
	handlePrevPos.x = prevPos.x;
	handlePrevPos.y = prevPos.y;

	if(ownPos != NULL)
	{
		handlePos.x = ownPos->x;
		handlePos.y = ownPos->y;
	}

	if(ownPrevPos != NULL)
	{
		handlePrevPos.x = ownPrevPos->x;
		handlePrevPos.y = ownPrevPos->y;
	}

	//get the new position
	HSVect2D newPos = CollisionReposition(gameObjects, result, &handlePos, &handlePrevPos);

	//if(

	//handle fighter-specific impact physics
	if(state != JUMPING && state != AIRBORN)
	{
		pos.x = newPos.x;
		pos.y = newPos.y;
	}
	else if(state == JUMPING)
	{
		//set the new velocity, states, and position
		if(result->horizontalImpact)
		{
			vel.x = 0;
			airVelocityCategory = LOW_VELOCITY;
		}
		if(result->verticalImpact)
		{
			if(vel.y > 0)
			{
				HandleJumpLanding();
				currentSurfaceFriction = result->lastTargetObject->friction;
			}
			vel.y = 0;
		}
		pos.x = newPos.x;
		pos.y = newPos.y;

		if(vel.x != 0 || vel.y != 0)
		{
			//get the remaining movement distance not covered in this frame
			HSVect2D checkDist;
			checkDist.x = 0;
			checkDist.y = 0;

			if(vel.x != 0)
			{
				checkDist.x = (handlePos.x - handlePrevPos.x) - (pos.x - handlePrevPos.x);
			}
			if(vel.y != 0)
			{
				checkDist.y = (handlePos.y - handlePrevPos.y) - (pos.y - handlePrevPos.y);
			}

			if(checkDist.x == 0 && checkDist.y == 0)
			{
				return 0;
			}

			//do a collision check all over again, using the new position as the previous position, and that position plus the remaining movement distance as the current position
			HSVect2D checkPos;
			checkPos.x = pos.x + checkDist.x;
			checkPos.y = pos.y + checkDist.y;

			TerrainCollisionResult result = IsCollidingWithAnyTerrain(gameObjects, NULL, &checkPos, &pos);
			if(result.lastSelfBox != NULL)
			{
				HandleTerrainCollision(gameObjects, &result, &checkPos, &pos);
			}
			else
			{
				pos.x = checkPos.x;
				pos.y = checkPos.y;
			}
		}
	}
	else if(state == AIRBORN)
	{
		//handle terrain impact physics like this is a basic physics object
		CollisionPhysics(&newPos, result, &handlePos, &handlePrevPos);
	}

	return 0;
}

int Fighter::StepCheck(list<HSObject*> * gameObjects)
{
	HSVect2D checkPos;
	checkPos.x = pos.x;
	checkPos.y = pos.y + stepHeight;

	TerrainCollisionResult result = IsCollidingWithAnyTerrain(gameObjects, NULL, &checkPos, &pos);

	if(!result.verticalImpact)
	{
		if(state == RUNNING)
		{
			airVelocityCategory = HIGH_VELOCITY_FORWARD;
		}
		ChangeHold(fighterEventHolds.jumpNeutralFall);
		state = JUMPING;
		falls = true;
	}
	else
	{
		//save the friction of the surface currently stood upon
		currentSurfaceFriction = result.lastTargetObject->friction;
	}

	return 0;
}

int Fighter::HandleJumpLanding()
{
	curAirActions = airActions;
	falls = false;
	attacking = false;

	if(airVelocityCategory != LOW_VELOCITY)
	{
		sliding = true;
	}
	else
	{
		vel.x = 0;
	}

	if(turnUponLanding)
	{
		if(facing == LEFT)
		{
			facing = RIGHT;
			hFlip = false;
		}
		else
		{
			facing = LEFT;
			hFlip = true;
		}
	}

	switch(landingAction)
	{
		case NO_LANDING_ACTION:
			if(blocking || curBlockstun > 0)
			{
				ChangeHold(fighterEventHolds.blockHigh);
			}
			else if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
			{
				ChangeHold(fighterEventHolds.runningStop);
				runStopping = true;
				timeSinceRunStop = 0;
			}
			else if(airVelocityCategory == HIGH_VELOCITY_BACKWARD)
			{
				ChangeHold(fighterEventHolds.jumpNeutralLand);
				runStopping = true;
			}
			else
			{
				ChangeHold(fighterEventHolds.jumpNeutralLand);
			}
			state = STANDING;
			lightNeutralAttack = false;
			jumpCancellable = false;
			break;
		case MOVE:
			if(blocking || curBlockstun > 0)
			{
				ChangeHold(fighterEventHolds.blockHigh);
				state = STANDING;
			}
			else if(airVelocityCategory == HIGH_VELOCITY_FORWARD)
			{
				if(turnUponLanding)
				{
					ChangeHold(fighterEventHolds.runningTurn);
					state = STANDING;
					turning = true;
					runAfterTurning = true;
				}
				else
				{
					ChangeHold(fighterEventHolds.running);
					vel.x = runSpeed * facing;
					state = RUNNING;
				}
			}
			else if(airVelocityCategory == HIGH_VELOCITY_BACKWARD)
			{
				if(turnUponLanding)
				{
					ChangeHold(fighterEventHolds.turn);
					state = STANDING;
					turning = true;
				}
				else
				{
					ChangeHold(fighterEventHolds.jumpNeutralLand);
					runStopping = true;
					state = STANDING;
				}
			}
			else
			{
				if(turnUponLanding)
				{
					ChangeHold(fighterEventHolds.turn);
					state = STANDING;
					turning = true;
				}
				else
				{
					ChangeHold(fighterEventHolds.walking);
					vel.x = walkSpeed * facing;
					state = WALKING;
				}
			}
			lightNeutralAttack = false;
			jumpCancellable = false;
			break;
		case DASH:
			if(airVelocityCategory == HIGH_VELOCITY_BACKWARD)
			{
				if(turnUponLanding)
				{
					ChangeHold(fighterEventHolds.turn);
					state = STANDING;
					turning = true;
					runAfterTurning = true;
				}
				else
				{
					ChangeHold(fighterEventHolds.jumpNeutralLand);
					runStopping = true;
					runAfterTurning = true;
					state = STANDING;
				}
			}
			else
			{
				if(turnUponLanding)
				{
					ChangeHold(fighterEventHolds.runningTurn);
					state = STANDING;
					turning = true;
					runAfterTurning = true;
				}
				else
				{
					ChangeHold(fighterEventHolds.running);
					vel.x = runSpeed * facing;
					state = RUNNING;
				}
			}
			lightNeutralAttack = false;
			jumpCancellable = false;
			break;
		case CROUCH:
			if(blocking || curBlockstun > 0)
			{
				ChangeHold(fighterEventHolds.blockLow);
			}
			else
			{
				ChangeHold(fighterEventHolds.crouch);
			}
			state = CROUCHING;
			lightNeutralAttack = false;
			jumpCancellable = false;
			break;
		case JUMP:
			//ChangeHold(fighterEventHolds.jumpNeutralStart);
			break;
		case NEUTRAL_LIGHT:
			ChangeHold(fighterEventHolds.attackLightNeutralGround);
			GroundAttackExecuted();
			break;
		case UP_LIGHT:
			ChangeHold(fighterEventHolds.attackLightUpGround);
			GroundAttackExecuted();
			break;
		case DOWN_LIGHT:
			ChangeHold(fighterEventHolds.attackLightDownGround);
			GroundAttackExecuted();
			state = CROUCHING;
			break;
		case FORWARD_LIGHT:
			ChangeHold(fighterEventHolds.attackLightForwardGround);
			GroundAttackExecuted();
			break;
	}

	return 0;
}

int Fighter::CollideTerrain(list<HSObject*> * gameObjects)
{
	//check for collisions with terrain, if this object has any terrain boxes
	if(firstTerrainBox != NULL)
	{
		//handle a change in terrain box
		HandleChangeInTerrainBoxes(gameObjects);

		//check for collisions, if this object is in motion
		if(vel.x != 0 || vel.y != 0)
		{
			TerrainCollisionResult result = IsCollidingWithAnyTerrain(gameObjects);
			if(result.lastSelfBox != NULL) { HandleTerrainCollision(gameObjects, &result); }
		}

		//check directly below this object if it is grounded. If there is no terrain there, either step down or start falling
		if(state != JUMPING && state != AIRBORN && state != AIR_DASHING)
		{
			StepCheck(gameObjects);
		}
		else
		{
			//set this to false since this object is now airborn, and guaranteed to have fallen inside of any jump-through terrain
			//that it was previously resting on
			ignoreJumpThroughTerrain = false;
		}
	}
	
	return 0;
}

void Fighter::ApplyAttackResults()
{
	if(!attackResults.struck) { return; }

	if(attackResults.hFlip) { facing = LEFT; hFlip = true; }
	else { facing = RIGHT; hFlip = false; }

	if(health > 0)
	{
		curHealth -= attackResults.damage;
	
		if(curHealth < 0)
		{
			curHealth = 0;
		}
		else if(curHealth > health) { curHealth = health; }
	}

	hud->healthMeter->SetValue((float)curHealth / (float)health);

	hitstunBlockability = attackResults.blockability;
	curHitstun = attackResults.hitstun;
	curBlockstun = attackResults.blockstun;
	vel.x = attackResults.force.x * -facing;
	vel.y = attackResults.force.y;

	if(vel.y < 0)
	{
		state = JUMPING;
		falls = true;
	}

	if((state == STANDING || state == CROUCHING) && vel.x != 0 && vel.y == 0)
	{
		sliding = true;
	}
	else
	{
		sliding = false;
	}

	if(curHitstun > 0)
	{
		if(state == STANDING || state == WALKING || state == RUNNING)
		{
			if(hitstunBlockability == MID || hitstunBlockability == UNBLOCKABLE)
			{
				ChangeHold(fighterEventHolds.hitstunLightMidStanding);
			}
			else if(hitstunBlockability == HIGH)
			{
				ChangeHold(fighterEventHolds.hitstunLightHighStanding);
			}
			else if(hitstunBlockability == LOW)
			{
				ChangeHold(fighterEventHolds.hitstunLightLowStanding);
			}
			state = STANDING;
		}
		else if(state == CROUCHING)
		{
			if(hitstunBlockability == MID || hitstunBlockability == UNBLOCKABLE || hitstunBlockability == HIGH)
			{
				ChangeHold(fighterEventHolds.hitstunLightMidCrouching);
			}
			else
			{
				ChangeHold(fighterEventHolds.hitstunLightLowCrouching);
			}
		}
		else if(state == JUMPING)
		{
			ChangeHold(fighterEventHolds.hitstunLightAir);
		}

		blocking = false;
	}
	else if(curBlockstun > 0)
	{
		if(state == STANDING || state == WALKING || state == RUNNING)
		{
			ChangeHold(fighterEventHolds.blockHigh);
			state = STANDING;
		}
		else if(state == CROUCHING)
		{
			ChangeHold(fighterEventHolds.blockLow);
		}
		else if(state == JUMPING)
		{
			ChangeHold(fighterEventHolds.blockAir);
		}

		blocking = true;
	}

	ResetAttackResults();
}

void Fighter::HandleHurtCollision(TerrainObject * attacker)
{
	attackResults.struck = true;
	attackResults.blockability = attacker->blockability;

	FighterFacing reqBlockDirection = LEFT;
	attackResults.hFlip = true;
	if(pos.x < attacker->pos.x)
	{
		reqBlockDirection = RIGHT;
		attackResults.hFlip = false;
	}

	if(facing == reqBlockDirection && attacker->blockability != UNBLOCKABLE &&
		((attacker->blockability == MID && blocking) ||
		(state == JUMPING && blocking) ||
		(attacker->blockability == HIGH && state == STANDING && blocking) ||
		(attacker->blockability == LOW && state == CROUCHING && blocking)))
	{
		attackResults.blockstun = attacker->blockstun;
		attackResults.force.x = attacker->force.x;
		attackResults.force.y = 0;
		return;
	}

	attackResults.damage += attacker->damage;
	attackResults.hitstun = attacker->hitstun;
	attackResults.force.x = attacker->force.x;
	attackResults.force.y = attacker->force.y;
}

bool Fighter::IsFighter()
{
	return true;
}

FighterHold * Fighter::CreateNewHold()
{
	FighterHold * newHold = new FighterHold();

	return newHold;
}

int Fighter::SaveEventHolds(HSObjectHold * hold, XMLElement * eventHolds)
{
	if(int error = TerrainObject::SaveEventHolds(hold, eventHolds) != 0)
	{
		return error;
	}

	//check the attribute names, and put the hold pointers into the proper event hold slots.
	unsigned int eventHoldId;
	eventHolds->QueryUnsignedAttribute("standing", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.standing = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("turn", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.turn = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("walk", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.walk = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("walking", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.walking = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("walkingTurn", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.walkingTurn = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("run", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.run = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("running", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.running = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("runningTurn", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.runningTurn = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("runningStop", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.runningStop = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("crouch", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.crouch = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("crouching", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.crouching = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("crouchingTurn", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.crouchingTurn = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("stand", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.stand = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralStart", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralStart = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralStartAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralStartAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralRising", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralRising = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralFall", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralFall = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralFalling", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralFalling = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpNeutralLand", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpNeutralLand = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpBackwardRising", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpBackwardRising = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("jumpBackwardFall", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.jumpBackwardFall = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("airDashForward", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.airDashForward = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("airDashBackward", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.airDashBackward = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("blockHigh", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.blockHigh = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("blockLow", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.blockLow = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("blockAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.blockAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightHighStanding", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightHighStanding = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightMidStanding", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightMidStanding = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightLowStanding", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightLowStanding = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightMidCrouching", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightMidCrouching = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightLowCrouching", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightLowCrouching = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("hitstunLightAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.hitstunLightAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightNeutralGround", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightNeutralGround = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightDownGround", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightDownGround = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightUpGround", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightUpGround = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightForwardGround", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightForwardGround = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightNeutralAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightNeutralAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightDownAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightDownAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightUpAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightUpAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightForwardAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightForwardAir = (FighterHold*)hold; }
	
	eventHoldId = 0;
	eventHolds->QueryUnsignedAttribute("attackLightBackwardAir", &eventHoldId);
	if(hold->id == eventHoldId) { fighterEventHolds.attackLightBackwardAir = (FighterHold*)hold; }

	return 0;
}

int Fighter::AddUprightTerrainBox(HSBox * newUprightTerrainBox, XMLElement * definition)
{
	if(int error = DefineBox(definition, newUprightTerrainBox) != 0)
	{
		return error; //there was an error defining the box
	}

	if(firstUprightTerrainBox == NULL)
	{
		//just make the new terrain box the first terrain box
		firstUprightTerrainBox = newUprightTerrainBox;
		lastUprightTerrainBox = newUprightTerrainBox;
	}
	else
	{
		//put the new terrain box at the end of the list
		lastUprightTerrainBox->nextBox = newUprightTerrainBox;
		lastUprightTerrainBox = newUprightTerrainBox;
	}

	return 0;
}

int Fighter::AddCrouchingTerrainBox(HSBox * newCrouchingTerrainBox, XMLElement * definition)
{
	if(int error = DefineBox(definition, newCrouchingTerrainBox) != 0)
	{
		return error; //there was an error defining the box
	}

	if(firstCrouchingTerrainBox == NULL)
	{
		//just make the new terrain box the first terrain box
		firstCrouchingTerrainBox = newCrouchingTerrainBox;
		lastCrouchingTerrainBox = newCrouchingTerrainBox;
	}
	else
	{
		//put the new terrain box at the end of the list
		lastCrouchingTerrainBox->nextBox = newCrouchingTerrainBox;
		lastCrouchingTerrainBox = newCrouchingTerrainBox;
	}

	return 0;
}

int Fighter::AddProneTerrainBox(HSBox * newProneTerrainBox, XMLElement * definition)
{
	if(int error = DefineBox(definition, newProneTerrainBox) != 0)
	{
		return error; //there was an error defining the box
	}

	if(firstProneTerrainBox == NULL)
	{
		//just make the new terrain box the first terrain box
		firstProneTerrainBox = newProneTerrainBox;
		lastProneTerrainBox = newProneTerrainBox;
	}
	else
	{
		//put the new terrain box at the end of the list
		lastProneTerrainBox->nextBox = newProneTerrainBox;
		lastProneTerrainBox = newProneTerrainBox;
	}

	return 0;
}

int Fighter::AddCompactTerrainBox(HSBox * newCompactTerrainBox, XMLElement * definition)
{
	if(int error = DefineBox(definition, newCompactTerrainBox) != 0)
	{
		return error; //there was an error defining the box
	}

	if(firstCompactTerrainBox == NULL)
	{
		//just make the new terrain box the first terrain box
		firstCompactTerrainBox = newCompactTerrainBox;
		lastCompactTerrainBox = newCompactTerrainBox;
	}
	else
	{
		//put the new terrain box at the end of the list
		lastCompactTerrainBox->nextBox = newCompactTerrainBox;
		lastCompactTerrainBox = newCompactTerrainBox;
	}

	return 0;
}

bool Fighter::ChangeHold(HSObjectHold* hold)
{
	return TerrainObject::ChangeHold(hold);
}

HSObjectHold * Fighter::GetDefaultHold()
{
	turning = false;
	runStopping = false;
	jumpStartup = false;
	attacking = false;
	lightNeutralAttack = false;
	jumpCancellable = false;
	victims.clear();
	hitAudioList.clear();
	blockability = MID;
	horizontalDirectionBasedBlock = false;
	reversedHorizontalBlock = false;
	damage = 0;
	hitstun = 0;
	blockstun = 0;
	force.x = 0;
	force.y = 0;
	trips = false;

	if(curHitstun > 0)
	{
		if(state == STANDING || state == WALKING || state == RUNNING)
		{
			if(hitstunBlockability == MID || hitstunBlockability == UNBLOCKABLE)
			{
				return fighterEventHolds.hitstunLightMidStanding;
			}
			else if(hitstunBlockability == HIGH)
			{
				return fighterEventHolds.hitstunLightHighStanding;
			}
			else if(hitstunBlockability == LOW)
			{
				return fighterEventHolds.hitstunLightLowStanding;
			}
			state = STANDING;
		}
		else if(state == CROUCHING)
		{
			if(hitstunBlockability == MID || hitstunBlockability == UNBLOCKABLE || hitstunBlockability == HIGH)
			{
				return fighterEventHolds.hitstunLightMidCrouching;
			}
			else
			{
				return fighterEventHolds.hitstunLightLowCrouching;
			}
		}
		else if(state == JUMPING)
		{
			return fighterEventHolds.hitstunLightAir;
		}
	}
	else if(curBlockstun > 0 || blocking)
	{
		if(state == STANDING || state == WALKING || state == RUNNING)
		{
			state = STANDING;
			return fighterEventHolds.blockHigh;
		}
		else if(state == CROUCHING)
		{
			return fighterEventHolds.blockLow;
		}
		else if(state == JUMPING)
		{
			return fighterEventHolds.blockAir;
		}
	}

	if(state == STANDING)
	{
		return fighterEventHolds.standing;
	}
	else if(state == WALKING)
	{
		return fighterEventHolds.walking;
	}
	else if(state == RUNNING)
	{
		return fighterEventHolds.running;
	}
	else if(state == CROUCHING)
	{
		return fighterEventHolds.crouching;
	}
	else if(state == JUMPING)
	{
		if(vel.y < 0)
		{
			if((facing == LEFT && vel.x > 0) || (facing == RIGHT && vel.x < 0))
			{
				return fighterEventHolds.jumpBackwardRising;
			}
			else
			{
				return fighterEventHolds.jumpNeutralRising;
			}
		}
		else
		{
			return fighterEventHolds.jumpNeutralFalling;
		}
	}
	else if(state == AIR_DASHING)
	{
		if(airDash == FORWARD_AIR_DASH)
		{
			return fighterEventHolds.airDashForward;
		}
		else if(airDash == BACKWARD_AIR_DASH)
		{
			return fighterEventHolds.airDashBackward;
		}
	}

	//default to whatever the base class does
	return TerrainObject::GetDefaultHold();
}

void Fighter::GroundAttackExecuted()
{
	walkAfterTurning = false;
	runAfterTurning = false;
	turning = false;
	runStopping = false;
	jump = NO_JUMP;
	blocking = false;
	attacking = true;
	victims.clear();
	if(state == RUNNING)
	{
		sliding = true;
	}
	state = STANDING;
}

void Fighter::AirAttackExecuted()
{
	blocking = false;
	attacking = true;
	victims.clear();
}

bool Fighter::ForwardWasTapped(InputStates * inputHistory, int frame, int framesAgo)
{
	if(inputHistory == NULL) { return false; }

	InputStates * curHistory = inputHistory;
	bool releaseFound = false;
	unsigned int curPastFrame = frame - curHistory->frame;

	//if there was input this frame, it doesn't count
	if(curPastFrame == 0)
	{
		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}

	while(curPastFrame <= framesAgo)
	{
		if((facing == LEFT && (curHistory->bKeyLeft.released || curHistory->bButtonLeft.released || curHistory->bHatLeft.released || curHistory->bStickLeft.released))
			|| (facing == RIGHT && (curHistory->bKeyRight.released || curHistory->bButtonRight.released || curHistory->bHatRight.released || curHistory->bStickRight.released)))
		{
			releaseFound = true;
		}

		if(releaseFound && ((facing == LEFT && (curHistory->bKeyLeft.pressed || curHistory->bButtonLeft.pressed || curHistory->bHatLeft.pressed || curHistory->bStickLeft.pressed))
			|| (facing == RIGHT && (curHistory->bKeyRight.pressed || curHistory->bButtonRight.pressed || curHistory->bHatRight.pressed || curHistory->bStickRight.pressed))))
		{
			return true;
		}

		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}

	return false;
}

bool Fighter::BackwardWasTapped(InputStates * inputHistory, int frame, int framesAgo)
{
	if(inputHistory == NULL) { return false; }

	InputStates * curHistory = inputHistory;
	bool releaseFound = false;
	unsigned int curPastFrame = frame - curHistory->frame;

	//if there was input this frame, it doesn't count
	if(curPastFrame == 0)
	{
		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}
	
	while(curPastFrame <= framesAgo)
	{
		if((facing == LEFT && (curHistory->bKeyRight.released || curHistory->bButtonRight.released || curHistory->bHatRight.released || curHistory->bStickRight.released))
			|| (facing == RIGHT && (curHistory->bKeyLeft.released || curHistory->bButtonLeft.released || curHistory->bHatLeft.released || curHistory->bStickLeft.released)))
		{
			releaseFound = true;
		}

		if(releaseFound && ((facing == LEFT && (curHistory->bKeyRight.pressed || curHistory->bButtonRight.pressed || curHistory->bHatRight.pressed || curHistory->bStickRight.pressed))
			|| (facing == RIGHT && (curHistory->bKeyLeft.pressed || curHistory->bButtonLeft.pressed || curHistory->bHatLeft.pressed || curHistory->bStickLeft.pressed))))
		{
			return true;
		}

		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}

	return false;
}

bool Fighter::ForwardHeld(InputStates * inputHistory)
{
	if(inputHistory == NULL) { return false; }

	if((facing == LEFT && (inputHistory->bKeyLeft.held || inputHistory->bButtonLeft.held || inputHistory->bHatLeft.held || inputHistory->bStickLeft.held)) ||
		(facing == RIGHT && (inputHistory->bKeyRight.held || inputHistory->bButtonRight.held || inputHistory->bHatRight.held || inputHistory->bStickRight.held)))
	{
		return true;
	}

	return false;
}

bool Fighter::BackwardHeld(InputStates * inputHistory)
{
	if(inputHistory == NULL) { return false; }

	if((facing == RIGHT && (inputHistory->bKeyLeft.held || inputHistory->bButtonLeft.held || inputHistory->bHatLeft.held || inputHistory->bStickLeft.held)) ||
		(facing == LEFT && (inputHistory->bKeyRight.held || inputHistory->bButtonRight.held || inputHistory->bHatRight.held || inputHistory->bStickRight.held)))
	{
		return true;
	}

	return false;
}

bool Fighter::ForwardHardPressed(InputStates * inputHistory, int frame, int framesAgo)
{
	if(inputHistory == NULL || inputHistory->frame != frame)
	{
		return false;
	}

	if((facing == LEFT && !inputHistory->bStickHardLeft.pressed) ||
		(facing == RIGHT && !inputHistory->bStickHardRight.pressed))
	{
		return false;
	}

	InputStates * curHistory = inputHistory;
	unsigned int curPastFrame = frame - curHistory->frame;

	while(curPastFrame <= framesAgo)
	{
		if(!curHistory->bStickUp.held && !curHistory->bStickDown.held && !curHistory->bStickLeft.held && !curHistory->bStickRight.held)
		{
			return true;
		}

		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}

	return false;
}

bool Fighter::BackwardHardPressed(InputStates * inputHistory, int frame, int framesAgo)
{
	if(inputHistory == NULL || inputHistory->frame != frame)
	{
		return false;
	}

	if((facing == LEFT && !inputHistory->bStickHardRight.pressed) ||
		(facing == RIGHT && !inputHistory->bStickHardLeft.pressed))
	{
		return false;
	}

	InputStates * curHistory = inputHistory;
	unsigned int curPastFrame = frame - curHistory->frame;
	
	while(curPastFrame <= framesAgo)
	{
		if(!curHistory->bStickUp.held && !curHistory->bStickDown.held && !curHistory->bStickLeft.held && !curHistory->bStickRight.held)
		{
			return true;
		}

		curHistory = curHistory->prevInputState;
		if(curHistory == NULL) { return false; }

		curPastFrame = frame - curHistory->frame;
	}

	return false;
}